<?php

namespace Snog\Forms\Admin\Controller;

use Snog\Forms\Entity\Form;
use Snog\Forms\Entity\Question;
use XF\Admin\Controller\AbstractController;
use XF\Html\Renderer\BbCode;
use XF\Mvc\ParameterBag;

class Forms extends AbstractController
{
	/**
	 * @param $action
	 * @param ParameterBag $params
	 * @return void
	 * @throws \XF\Mvc\Reply\Exception
	 */
	protected function preDispatchController($action, ParameterBag $params)
	{
		$this->assertAdminPermission('snogFormsAdmin');
	}

	public function actionMain()
	{
		$this->setSectionContext('snog_forms');
		return $this->view('Snog:Forms', 'snog_forms_main');
	}

	public function actionIndex()
	{
		$forms = $this->finder('Snog\Forms:Form')
			->order('display')
			->fetch();

		$viewParams = [
			'forms' => $forms
		];
		return $this->view('Snog:Forms\Forms', 'snog_forms_form_list', $viewParams);
	}

	public function actionAdd()
	{
		/** @var Form $form */
		$form = $this->em()->create('Snog\Forms:Form');
		$formCount = 0;

		/** @var Form $lastForm */
		$lastForm = $this->finder('Snog\Forms:Form')->order('display')->fetchOne();
		if ($lastForm)
		{
			$formCount = $lastForm->display + 1;
		}

		return $this->formAddEdit($form, $formCount);
	}

	public function formAddEdit(Form $form, $formDisplay = 0)
	{
		/** @var \XF\Repository\Moderator $moderatorRepo */
		$moderatorRepo = $this->repository('XF:Moderator');
		$moderatorPermissionData = $moderatorRepo->getModeratorPermissionData('node');

		$data['start'] = date('Y', time());
		$data['end'] = $data['start'] + 25;
		$data['hours'] = ['00', '01', '02', '03', '04', '05', '06', '07', '08', '09', 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23];
		$data['minutes'] = ['00', '05', 10, 15, 20, 25, 30, 35, 40, 45, 50, 55];
		$data['starthour'] = '00';
		$data['startminute'] = '00';
		$data['endhour'] = '00';
		$data['endminute'] = '00';

		// SET NEXT DISPLAY VALUE IF NEW FORM
		if (!$form->display)
		{
			$form->display = $formDisplay;
		}

		// TIMEZONE TO GUEST TIMEZONE
		// ENSURES FORM STARTS/STOPS AT THE SAME TIME FOR EVERYONE
		date_default_timezone_set($this->options()->guestTimeZone);

		if ($form->start)
		{
			$data['starthour'] = date('H', $form->start);
			$data['startminute'] = date('i', $form->start);
		}

		if ($form->end)
		{
			$data['endhour'] = date('H', $form->end);
			$data['endminute'] = date('i', $form->end);
		}

		/** @var \XF\Repository\Style $styleRepo */
		$styleRepo = $this->repository('XF:Style');

		/** @var \XF\Repository\Node $nodeRepo */
		$nodeRepo = $this->repository('XF:Node');

		/** @var \Snog\Forms\Repository\Type $typeRepo */
		$typeRepo = $this->repository('Snog\Forms:Type');

		/** @var \XF\Repository\UserGroup $userGroupRepo */
		$userGroupRepo = $this->repository('XF:UserGroup');

		$userCriteria = $this->app->criteria('XF:User', $form->user_criteria);

		/** @var Question[] $originalQuestions */
		$questions = $this->finder('Snog\Forms:Question')->where('posid', $form->posid)->fetch();

		$paymentRepo = $this->repository('XF:Payment');
		$paymentProfiles = $paymentRepo->findPaymentProfilesForList()->fetch();

		$viewParams = [
			'form' => $form,
			'questions' => $questions,
			'data' => $data,
			'types' => $typeRepo->findTypesForList()->fetch(),
			'userGroups' => $userGroupRepo->findUserGroupsForList()->fetch(),
			'nodeTree' => $nodeRepo->createNodeTree($nodeRepo->getFullNodeList()),
			'availablePrefixes' => $this->getPrefixes(),
			'interfaceGroups' => $moderatorPermissionData['interfaceGroups'],
			'globalPermissions' => $moderatorPermissionData['globalPermissions'],
			'contentPermissions' => $moderatorPermissionData['contentPermissions'],
			'userCriteria' => $userCriteria,
			'styleTree' => $styleRepo->getStyleTree(false),
			'profiles' => $paymentProfiles,
			'totalProfiles' => $paymentProfiles->count(),
		];

		return $this->view('Snog:Forms\Forms', 'snog_forms_form_edit', $viewParams);
	}

	/**
	 * @param ParameterBag $params
	 * @return \XF\Mvc\Reply\View
	 * @throws \XF\Mvc\Reply\Exception
	 */
	public function actionEdit(ParameterBag $params)
	{
		$form = $this->assertFormExists($params['posid']);
		return $this->formAddEdit($form);
	}

	public function actionCopy(ParameterBag $params)
	{
		$formCount = 0;
		$forms = $this->finder('Snog\Forms:Form')->order('display')->fetch();

		/** @var Form $lastForm */
		$lastForm = $forms ? $forms->last() : null;

		if ($lastForm)
		{
			$formCount = $lastForm->display + 1;
		}

		/** @var Form $form */
		$form = $this->em()->find('Snog\Forms:Form', $params['posid']);

		/** @var \XF\Repository\Moderator $moderatorRepo */
		$moderatorRepo = $this->repository('XF:Moderator');
		$moderatorPermissionData = $moderatorRepo->getModeratorPermissionData('node');

		$form->set('posid', 0, ['forceSet' => true]);
		$form->position = 'Copy Of: ' . $form->position;
		$form->active = false;
		$form->display = $formCount;

		/** @var \XF\Repository\Style $styleRepo */
		$styleRepo = $this->repository('XF:Style');

		/** @var \XF\Repository\Node $nodeRepo */
		$nodeRepo = $this->repository('XF:Node');

		/** @var \Snog\Forms\Repository\Type $typeRepo */
		$typeRepo = $this->repository('Snog\Forms:Type');

		/** @var \XF\Repository\UserGroup $userGroupRepo */
		$userGroupRepo = $this->repository('XF:UserGroup');

		$userCriteria = $this->app->criteria('XF:User', $form->user_criteria);

		/** @var Question[] $originalQuestions */
		$questions = $this->finder('Snog\Forms:Question')->where('posid', $form->posid)->fetch();

		$viewParams = [
			'form' => $form,
			'questions' => $questions,
			'action' => 'edit',
			'copyform' => $params['posid'],
			'types' => $typeRepo->findTypesForList()->fetch(),
			'userGroups' => $userGroupRepo->findUserGroupsForList()->fetch(),
			'nodeTree' => $nodeRepo->createNodeTree($nodeRepo->getFullNodeList()),
			'availablePrefixes' => $this->getPrefixes(),
			'interfaceGroups' => $moderatorPermissionData['interfaceGroups'],
			'globalPermissions' => $moderatorPermissionData['globalPermissions'],
			'contentPermissions' => $moderatorPermissionData['contentPermissions'],
			'userCriteria' => $userCriteria,
			'styleTree' => $styleRepo->getStyleTree(false)
		];

		return $this->view('Snog:Forms\Forms', 'snog_forms_form_edit', $viewParams);
	}

	/**
	 * @param ParameterBag $params
	 * @return \XF\Mvc\Reply\Redirect
	 * @throws \XF\Mvc\Reply\Exception
	 * @throws \XF\PrintableException
	 */
	public function actionSave(ParameterBag $params)
	{
		if ($params->posid)
		{
			$modifiedForm = $this->assertFormExists($params->posid);
		}
		else
		{
			/** @var Form $modifiedForm */
			$modifiedForm = $this->em()->create('Snog\Forms:Form');
		}

		$this->formSaveProcess($modifiedForm)->run();

		$copyId = $this->filter('copyform', 'uint');

		// FORM IS BEING COPIED - COPY QUESTIONS FROM THE ORIGINAL FORM
		if ($copyId)
		{
			/** @var Question[] $originalQuestions */
			$originalQuestions = $this->finder('Snog\Forms:Question')->where('posid', $copyId)->fetch();

			$newParents = [];

			foreach ($originalQuestions as $originalQuestion)
			{
				/** @var Question $newQuestion */
				$newQuestion = $this->em()->create('Snog\Forms:Question');

				$newQuestion->posid = $modifiedForm->posid;
				$newQuestion->text = $originalQuestion->text;
				$newQuestion->description = $originalQuestion->description;
				$newQuestion->type = $originalQuestion->type;
				$newQuestion->error = $originalQuestion->error;
				$newQuestion->expected = $originalQuestion->expected;
				$newQuestion->display = $originalQuestion->display;
				$newQuestion->display_parent = $originalQuestion->display_parent;
				$newQuestion->regex = $originalQuestion->regex;
				$newQuestion->regexerror = $originalQuestion->regexerror;
				$newQuestion->defanswer = $originalQuestion->defanswer;
				$newQuestion->questionpos = $originalQuestion->questionpos;
				$newQuestion->showquestion = $originalQuestion->showquestion;
				$newQuestion->showunanswered = $originalQuestion->showunanswered;
				$newQuestion->inline = $originalQuestion->inline;
				$newQuestion->format = $originalQuestion->format;
				$newQuestion->placeholder = $originalQuestion->placeholder;

				if ($originalQuestion->hasconditional)
				{
					$newQuestion->hasconditional = $originalQuestion->hasconditional;
				}

				// ASSIGN NEW PARENT ID TO CHILD
				if ($originalQuestion->conditional)
				{
					foreach ($newParents as $newParent)
					{
						if ($newParent['oldId'] == $originalQuestion->conditional)
						{
							$newQuestion->conditional = $newParent['newId'];
						}
					}
				}

				$newQuestion->conanswer = $originalQuestion->conanswer;
				$newQuestion->save();

				// BUILD LIST OF OLD PARENT QUESTIONS
				if ($originalQuestion->hasconditional)
				{
					$newParents[] = [
						'oldId' => $originalQuestion->questionid,
						'newId' => $newQuestion->getEntityId()
					];
				}
			}

			// GET PARENT QUESTIONS AND REBUILD CHILD LIST
			if (!empty($newParents))
			{
				$findQuestions = '';

				foreach ($newParents as $newParent)
				{
					if ($findQuestions) $findQuestions .= ',';
					$findQuestions .= $newParent['newId'];
				}

				$questionFinder = $this->finder('Snog\Forms:Question');
				$questionFinder->whereSql(
					"FIND_IN_SET(" . $questionFinder->columnSqlName('questionid') . ", "
					. $questionFinder->quote($findQuestions) . ")"
				);

				/** @var Question[] $changeParents */
				$changeParents = $questionFinder->fetch();

				foreach ($changeParents as $changeParent)
				{
					$newChildList = [];

					/** @var Question[] $newChildren */
					$newChildren = $this->finder('Snog\Forms:Question')
						->where('conditional', $changeParent->questionid)
						->fetch();

					foreach ($newChildren as $newChild)
					{
						$newChildList[] = $newChild->questionid;
					}

					$changeParent->hasconditional = $newChildList;
					$changeParent->save();
				}
			}
		}

		return $this->redirect($this->buildLink('form-forms'));
	}

	/**
	 * @param Form $modifiedForm
	 * @return \XF\Mvc\FormAction
	 * @throws \XF\Mvc\Reply\Exception
	 * @throws \XF\PrintableException
	 */
	protected function formSaveProcess(Form $modifiedForm)
	{
		$form = $this->formAction();
		$input = $this->filter([
			'position' => 'str',
			'node_id' => 'uint',
			'secnode_id' => 'uint',
			'active' => 'bool',
			'subject' => 'str',
			'email' => 'str',
			'email_attachments' => 'bool',
			'email_parent' => 'uint',
			'inthread' => 'bool',
			'insecthread' => 'bool',
			'posterid' => 'str',
			'secposterid' => 'str',
			'bypm' => 'bool',
			'pmsender' => 'str',
			'pmdelete' => 'bool',
			'pmerror' => 'str',
			'pmto' => 'str',
			'appid' => 'uint',
			'returnto' => 'uint',
			'returnlink' => 'str',
			'postapproval' => 'bool',
			'parseyesno' => 'bool',
			'incname' => 'bool',
			'oldthread' => 'uint',
			'pmapp' => 'bool',
			'pmtext' => 'str',
			'confirmation_email' => 'bool',
			'confirmation_email_question_id' => 'uint',
			'confirmation_email_title' => 'str',
			'confirmation_email_text' => 'str',
			'confirmation_email_attachments' => 'bool',
			'apppromote' => 'bool',
			'promoteto' => 'uint',
			'appadd' => 'bool',
			'addto' => 'array',
			'postpoll' => 'bool',
			'pollpublic' => 'bool',
			'pollchange' => 'bool',
			'pollview' => 'bool',
			'pollquestion' => 'str',
			'promote_type' => 'uint',
			'pollclose' => 'uint',
			'pollpromote' => 'array',
			'decidepromote' => 'uint',
			'removeinstant' => 'bool',
			'approved_title' => 'str',
			'approved_text' => 'str',
			'denied_title' => 'str',
			'denied_text' => 'str',
			'app_style' => 'uint',
			'user_criteria' => 'array',
			'watchthread' => 'bool',
			'make_moderator' => 'uint',
			'instant' => 'bool',
			'aboveapp' => 'str',
			'belowapp' => 'str',
			'approved_file' => 'str',
			'normalpoll' => 'bool',
			'normalpublic' => 'bool',
			'normalchange' => 'bool',
			'normalview' => 'bool',
			'normalclose' => 'uint',
			'normalquestion' => 'str',
			'threadapp' => 'bool',
			'threadbutton' => 'str',
			'post_thread_button_node_ids' => 'array-uint',
			'post_thread_button_replace' => 'bool',
			'thanks' => 'str',
			'formlimit' => 'uint',
			'response' => 'array',
			'qcolor' => 'str',
			'acolor' => 'str',
			'forummod' => 'array',
			'supermod' => 'array',
			'quickreply' => 'bool',
			'store' => 'bool',
			'start' => 'str',
			'end' => 'str',
			'qroption' => 'bool',
			'qrbutton' => 'str',
			'qrstarter' => 'bool',
			'qrforums' => 'array',
			'beforeerror' => 'str',
			'aftererror' => 'str',
			'bbstart' => 'str',
			'bbend' => 'str',
			'display' => 'uint',
			'minimum_attachments' => 'uint',
			'is_public_visible' => 'bool',
			'cooldown' => 'int',
			'cost_currency' => 'str',
			'cost_amount' => 'float',
			'payment_profile_ids' => 'array-uint',
			'extra_data' => 'array'
		]);

		$addOns = $this->app()->container('addon.cache');
		$isMultiPrefix = isset($addOns['SV/MultiPrefix']);

		$input['prefix_ids'] = $isMultiPrefix
			? $this->filter('prefix_ids', 'array-uint')
			: [$this->filter('prefix_id', 'uint')];

		// CHECK REPORT TITLE FOR QUESTION ANSWERS
		preg_match_all('/({A\d+})/', $input['subject'], $titleAnswers);
		$questionIds = [];

		if (!empty($titleAnswers[1]))
		{
			foreach ($titleAnswers[1] as $titleAnswer)
			{
				$questionNumber = str_replace('{A', '', $titleAnswer);
				$questionNumber = str_replace('}', '', $questionNumber);
				$questionIds[] = $questionNumber;
			}
		}

		// VERIFY QUESTIONS FOR TITLE ARE REQUIRED TO BE ANSWERED
		if (!empty($questionIds) && $modifiedForm->posid)
		{
			/** @var \XF\Mvc\Entity\AbstractCollection|Question[] $questions */
			$questions = $this->finder('Snog\Forms:Question')
				->where('posid', $modifiedForm->posid)
				->order('display', 'ASC')
				->fetch();

			$questionArray = $questions->toArray();

			if (!empty($questionArray))
			{
				$errors = [];

				// CHECK TO BE SURE QUESTIONS EXIST AND MEET REPORT TITLE REQUIREMENTS
				foreach ($questions as $question)
				{
					if (in_array($question->display, $questionIds))
					{
						// NOT A REQUIRED QUESTION - THAT'S AN ERROR
						if (!$question->error)
						{
							$errors[] = \XF::phrase('snog_forms_error_title_answer', ['number' => $question->display]);
						}

						// NOT A TYPE THAT CAN BE USED IN REPORT TITLE - THAT'S AN ERROR
						if (!$question->canUsedForReportTitle())
						{
							$errors[] = \XF::phrase('snog_forms_error_title_type', ['number' => $question->display]);
						}

						// REMOVE CURRENT QUESTION FROM TITLE SEARCH
						if (($key = array_search($question->display, $questionIds)) !== false)
						{
							unset($questionIds[$key]);
						}
					}
				}

				// NON-EXISTENT QUESTIONS ARE IN REPORT TITLE - THROW ERROR
				if (!empty($questionIds))
				{
					foreach ($questionIds as $questionId)
					{
						$errors[] = \XF::phrase('snog_forms_error_title_not_exist', ['number' => $questionId]);
					}
				}

				if ($errors)
				{
					throw $this->errorException($errors);
				}
			}
		}
		else
		{
			// NO QUESTIONS DEFINED BUT ANSWERS IN TITLE - THROW ERROR
			if (empty($questionIds) && !empty($titleAnswers[1]))
			{
				throw $this->errorException(\XF::phrase('snog_forms_error_title_assign'));
			}
		}

		$timevalues = $this->filter([
			'starthour' => 'str',
			'startminute' => 'str',
			'endhour' => 'str',
			'endminute' => 'str',
		]);

		// THREAD BUTTON REMOVED FROM FORM - REMOVE FROM FORUM NODE
		if ($modifiedForm->node_id && !$input['threadapp'])
		{
			$update = ['snog_posid' => 0, 'snog_label' => ""];
			$this->app->db()->update('xf_node', $update, 'snog_posid = ?', $modifiedForm->posid);
		}

		// FORM LIMIT REMOVED - RESET ALL USER COUNTS
		if ($modifiedForm->formlimit && !$input['formlimit'])
		{
			$userFinder = $this->finder('XF:User');

			/** @var \Snog\Forms\XF\Entity\User[] $users */
			$users = $userFinder
				->where('snog_forms', 'LIKE', $userFinder->escapeLike('"posid":' . $modifiedForm->posid . ',', '%?%'))
				->fetch();

			foreach ($users as $name)
			{
				$name->updateAdvancedFormsSerials($modifiedForm->posid);
				if ($name->isChanged('snog_forms'))
				{
					$name->save(false, false);
				}
			}
		}

		// THERE IS AN EXTRA RESPONSE SENT BY XF ITSELF - CLEAR IT
		foreach ($input['response'] as $key => $response)
		{
			if (!$response) unset($input['response'][$key]);
		}

		// TIMEZONE TO GUEST TIMEZONE
		// ENSURES FORM STARTS/STOPS AT THE SAME TIME FOR EVERYONE
		date_default_timezone_set($this->options()->guestTimeZone);

		if ($input['start'])
		{
			$input['start'] = strtotime($input['start'] . ' ' . $timevalues['starthour'] . ':' . $timevalues['startminute']);
		}

		if ($input['end'])
		{
			$input['end'] = strtotime($input['end'] . ' ' . $timevalues['endhour'] . ':' . $timevalues['endminute']);
		}

		$htmlInput = $this->filter([
			'aboveapp_html' => 'str',
			'belowapp_html' => 'str',
			'pmtext_html' => 'str',
			'approved_text_html' => 'str',
			'denied_text_html' => 'str',
			'confirmation_email_text_html' => 'str',
		]);

		if ($htmlInput['aboveapp_html'])
		{
			$input['aboveapp'] = BbCode::renderFromHtml($htmlInput['aboveapp_html']);
		}
		if ($htmlInput['belowapp_html'])
		{
			$input['belowapp'] = BbCode::renderFromHtml($htmlInput['belowapp_html']);
		}
		if ($htmlInput['pmtext_html'])
		{
			$input['pmtext'] = BbCode::renderFromHtml($htmlInput['pmtext_html']);
		}
		if ($htmlInput['approved_text_html'])
		{
			$input['approved_text'] = BbCode::renderFromHtml($htmlInput['approved_text_html']);
		}
		if ($htmlInput['denied_text_html'])
		{
			$input['denied_text'] = BbCode::renderFromHtml($htmlInput['denied_text_html']);
		}
		if ($htmlInput['confirmation_email_text_html'])
		{
			$input['confirmation_email_text'] = BbCode::renderFromHtml($htmlInput['confirmation_email_text_html']);
		}

		if (empty($input['user_criteria']['connected_accounts']['data']))
		{
			unset($input['user_criteria']['connected_accounts']);
		}

		foreach ($input['extra_data'] as $extraDataKey => $extraData)
		{
			if ($extraDataKey == 'email_condition_question_answer')
			{
				$conditionQuestionAnswerEmailData = $extraData['data'] ?? [];
				foreach ($conditionQuestionAnswerEmailData as $key => $conditionQuestionAnswerEmailDatum)
				{
					if (!isset($conditionQuestionAnswerEmailDatum['answer']) || empty($conditionQuestionAnswerEmailDatum['email']))
					{
						unset($input['extra_data']['email_condition_question_answer']['data'][$key]);
					}
				}
			}
		}



		$form->basicEntitySave($modifiedForm, $input);

		return $form;
	}

	public function actionAddDefault(ParameterBag $params)
	{
		/** @var Form $form */
		$form = $this->assertFormExists($params->posid);

		/** @var Question[] $defaultQuestions */
		$questionFinder = $this->finder('Snog\Forms:Question')
			->where('posid', '=', 0);

		$questionIds = $this->filter('question_ids', 'array-uint');
		if ($questionIds)
		{
			$questionFinder->where('questionid', $questionIds);
		}

		$defaultQuestions = $questionFinder->fetch();

		if ($this->isPost())
		{
			$conditionedQuestions = [];

			$maxExistingQuestionPos = $this->app->db()->fetchOne("
				SELECT MAX(display)
				FROM xf_snog_forms_questions
				WHERE posid = ?
			", $params->posid);

			foreach ($defaultQuestions as $question)
			{
				/** @var Question $modifiedQuestion */
				$modifiedQuestion = $this->em()->create('Snog\Forms:Question');

				$newQuestion = $question->getDataForCopy();

				$newQuestion['posid'] = $form->posid;

				// CHANGE PARENT FROM DEFAULT QUESTION TO NEW QUESTION
				if ($question->conditional)
				{
					foreach ($conditionedQuestions as $condition)
					{
						if ($condition['oldParent'] == $question->conditional)
						{
							$newQuestion['conditional'] = $condition['newParent'];
							break;
						}
					}
				}

				$maxExistingQuestionPos++;
				$newQuestion['display'] = $maxExistingQuestionPos;

				$this->questionAddDefaultProcess($modifiedQuestion, $newQuestion)->run();

				// ADD PARENT INFO TO ARRAY
				if ($question->hasconditional)
				{
					$conditionedQuestions[] = [
						'oldParent' => $question->questionid,
						'oldParentConditionals' => $question->hasconditional,
						'newParent' => $modifiedQuestion->questionid
					];
				}

				// ADD PROPER CONDITIONAL QUESTIONS TO PARENT
				if ($question->conditional)
				{
					foreach ($conditionedQuestions as $condition)
					{
						if ($condition['oldParent'] == $question->conditional)
						{
							/** @var Question $parentQuestion */
							$parentQuestion = $this->em()->findOne('Snog\Forms:Question', [
								'questionid', '=', $condition['newParent']
							]);
							if (!$parentQuestion)
							{
								continue;
							}

							/** @var Question $oldParentQuestion */
							$oldParentQuestion = $this->em()->findOne('Snog\Forms:Question', [
								'questionid', '=', $condition['oldParent']
							]);
							if (!$oldParentQuestion)
							{
								continue;
							}

							/** @var Question[] $childQuestions */
							$childQuestions = $this->finder('Snog\Forms:Question')
								->where('conditional', $parentQuestion->questionid)
								->fetch();

							$existingConditionals = [];
							if ($parentQuestion->hasconditional)
							{
								$existingConditionals = $parentQuestion->hasconditional;
							}

							// CREATE NEW CONDITIONALS LIST FOR NEW PARENT
							if (empty($existingConditionals) || $childQuestions)
							{
								foreach ($childQuestions as $childQuestion)
								{
									if (!in_array($childQuestion->questionid, $existingConditionals))
									{
										$existingConditionals[] = $childQuestion->questionid;
									}
								}
							}

							// REMOVE DEFAULT CONDITIONAL QUESTIONS FROM EXISTING CONDITIONAL LIST
							if ($oldParentQuestion->posid == 0)
							{
								foreach ($condition['oldParentConditionals'] as $oldConditional)
								{
									if (($key = array_search($oldConditional, $existingConditionals)) !== false)
									{
										unset($existingConditionals[$key]);
									}
								}
							}

							$parentQuestion->hasconditional = $existingConditionals;
							$parentQuestion->save();
						}
					}
				}
			}

			return $this->redirect($this->buildLink('form-editquestions/formquestions', $form));
		}

		$viewParams = [
			'form' => $form,
			'questions' => $defaultQuestions
		];
		return $this->view('Snog:Forms\Forms\AddDefault', 'snog_forms_form_default_questions_add', $viewParams);
	}

	protected function questionAddDefaultProcess($modifiedQuestion, $values)
	{
		$questionAddDefault = $this->formAction();
		$questionAddDefault->basicEntitySave($modifiedQuestion, $values);
		return $questionAddDefault;
	}

	/**
	 * @param ParameterBag $params
	 * @return \XF\Mvc\Reply\Redirect|\XF\Mvc\Reply\View
	 * @throws \XF\Mvc\Reply\Exception
	 * @throws \XF\PrintableException
	 */
	public function actionDelete(ParameterBag $params)
	{
		$form = $this->assertFormExists($params->posid);

		if ($this->isPost())
		{
			$form->delete();

			/** @var \Snog\Forms\Repository\Question $questionRepo */
			$questionRepo = $this->repository('Snog\Forms:Question');
			$questionRepo->deleteQuestions($form->posid);

			return $this->redirect($this->buildLink('form-forms'));
		}

		$viewParams = ['form' => $form];
		return $this->view('Snog:Forms\Forms', 'snog_forms_confirm', $viewParams);
	}

	/**
	 * @param ParameterBag $params
	 * @return \XF\Mvc\Reply\Redirect|\XF\Mvc\Reply\View
	 * @throws \XF\Mvc\Reply\Exception
	 * @throws \XF\PrintableException
	 */
	public function actionReset(ParameterBag $params)
	{
		$form = $this->assertFormExists($params->posid);

		if ($this->isPost())
		{
			$userFinder = $this->finder('XF:User');

			/** @var \Snog\Forms\XF\Entity\User[] $users */
			$users = $userFinder
				->where(
					'snog_forms',
					'LIKE',
					$userFinder->escapeLike('"posid":' . $form->posid . ',', '%?%')
				)
				->fetch();

			foreach ($users as $user)
			{
				/** @var \Snog\Forms\Repository\Log $logRepo */
				$logRepo = $this->repository('Snog\Forms:Log');
				$logRepo->resetCountedFormLogsForUser($form, $user);
			}

			return $this->redirect($this->buildLink('form-forms'), \XF::phrase('success'));
		}

		$viewParams = ['form' => $form];
		return $this->view('Snog\Forms:Forms\Reset', 'snog_forms_form_reset_confirm', $viewParams);
	}

	/**
	 * @param ParameterBag $params
	 * @return \XF\Mvc\Reply\AbstractReply|\XF\Mvc\Reply\Redirect|\XF\Mvc\Reply\View
	 * @throws \XF\Mvc\Reply\Exception
	 * @throws \XF\PrintableException
	 */
	public function actionResetOne(ParameterBag $params)
	{
		$form = $this->assertFormExists($params->posid);

		if ($this->isPost())
		{
			$input = $this->filter(['resetuser' => 'str']);

			/** @var \Snog\Forms\XF\Entity\User $user */
			$user = $this->em()->findOne('XF:User', ['username', '=', $input['resetuser']]);
			if (!$user)
			{
				return $this->notFound(\XF::phrase('unknown_user'));
			}

			/** @var \Snog\Forms\Repository\Log $logRepo */
			$logRepo = $this->repository('Snog\Forms:Log');
			$logRepo->resetCountedFormLogsForUser($form, $user);

			return $this->redirect($this->buildLink('form-forms'), \XF::phrase('success'));
		}

		$viewParams = ['form' => $form];
		return $this->view('Snog:Forms\Forms', 'snog_forms_form_reset_user_confirm', $viewParams);
	}

	/**
	 * @return \XF\Mvc\Reply\Redirect|\XF\Mvc\Reply\View
	 */
	public function actionSort()
	{
		$forms = $this->finder('Snog\Forms:Form')->order('display')->fetch();

		/** @var \Snog\Forms\Repository\Form $formRepo */
		$formRepo = $this->repository('Snog\Forms:Form');
		$formList = $formRepo->createFormTree($forms);

		if ($this->isPost())
		{
			/** @var \XF\ControllerPlugin\Sort $sorter */
			$sorter = $this->plugin('XF:Sort');

			$options = [
				'orderColumn' => 'display',
				'jump' => 1,
				'preSaveCallback' => null
			];

			$sortTree = $sorter->buildSortTree($this->filter('forms', 'json-array'));
			$sorter->sortTree($sortTree, $formList->getAllData(), 'display_parent', $options);

			return $this->redirect($this->buildLink('form-forms'));
		}

		$viewParams = ['formList' => $formList];
		return $this->view('Snog:Forms\Sort', 'snog_forms_form_order', $viewParams);
	}

	/**
	 * @param $id
	 * @param null $with
	 * @return Form|\XF\Mvc\Entity\Entity
	 * @throws \XF\Mvc\Reply\Exception
	 * @noinspection PhpReturnDocTypeMismatchInspection
	 */
	protected function assertFormExists($id, $with = null)
	{
		return $this->assertRecordExists('Snog\Forms:Form', $id, $with, 'snog_forms_form_not_found');
	}

	protected function getPrefixes()
	{
		$prefixFinder = $this->finder('XF:ThreadPrefix');
		$availablePrefixes = $prefixFinder->order('materialized_order')->fetch();
		return $availablePrefixes->pluckNamed('title', 'prefix_id');
	}

	/**
	 * @return \Snog\Forms\Repository\Question|\XF\Mvc\Entity\Repository
	 * @noinspection PhpReturnDocTypeMismatchInspection
	 */
	protected function getQuestionRepo()
	{
		return $this->repository('Snog\Forms:Question');
	}
}