<?php

namespace Snog\Forms\Admin\Controller;

use Snog\Forms\Entity\Form;
use Snog\Forms\Entity\Question;
use XF\Admin\Controller\AbstractController;
use XF\Mvc\FormAction;
use XF\Mvc\ParameterBag;

class Questions extends AbstractController
{
	/**
	 * @param $action
	 * @param ParameterBag $params
	 * @return void
	 * @throws \XF\Mvc\Reply\Exception
	 */
	protected function preDispatchController($action, ParameterBag $params)
	{
		$this->assertAdminPermission('snogFormsAdmin');
	}

	public function actionIndex()
	{
		$questionRepo = $this->getQuestionRepo();

		$questionTree = $questionRepo->createQuestionTree($questionRepo->getQuestionList());
		$conditionalPermitted = false;

		foreach ($questionTree as $tree)
		{
			/** @var Question $question */
			$question = $tree->record;
			if ($question->canTriggerConditionals())
			{
				$conditionalPermitted = true;
			}
		}

		$viewParams = [
			'questions' => $questionTree,
			'conditionalPermitted' => $conditionalPermitted
		];

		return $this->view('Snog:Forms\Questions', 'snog_forms_question_list', $viewParams);
	}

	public function actionFormQuestions(ParameterBag $params)
	{
		$questionRepo = $this->getQuestionRepo();

		$questionTree = $questionRepo->createQuestionTree($questionRepo->getQuestionList($params['posid']));
		$conditionalPermitted = false;

		foreach ($questionTree as $tree)
		{
			/** @var Question $question */
			$question = $tree->record;
			if ($question->canTriggerConditionals())
			{
				$conditionalPermitted = true;
			}
		}

		/** @var Form $editForm */
		$editForm = $this->em()->findOne('Snog\Forms:Form', ['posid', '=', $params['posid']]);
		$viewParams = [
			'questions' => $questionTree,
			'form' => $editForm,
			'conditionalPermitted' => $conditionalPermitted
		];

		return $this->view('Snog:Forms\Questions', 'snog_forms_question_list', $viewParams);
	}

	public function actionSort(ParameterBag $params)
	{
		$questionRepo = $this->getQuestionRepo();
		$questionList = $questionRepo->createQuestionTree($questionRepo->getQuestionList($params->posid ?: 0));

		if ($this->isPost())
		{
			/** @var \XF\ControllerPlugin\Sort $sorter */
			$sorter = $this->plugin('XF:Sort');

			$options = [
				'orderColumn' => 'display',
				'jump' => 1,
				'preSaveCallback' => null
			];

			$sortTree = $sorter->buildSortTree($this->filter('questions', 'json-array', 'posid'));
			$sorter->sortTree($sortTree, $questionList->getAllData(), 'display_parent', $options);

			if ($params->posid)
			{
				return $this->redirect($this->buildLink('form-editquestions/formquestions', $params));
			}

			return $this->redirect($this->buildLink('form-questions'));
		}

		$viewParams = [
			'questionList' => $questionList,
			'posid' => $params->posid,
			'form' => $params
		];

		return $this->view('Snog:Forms\Sort', 'snog_forms_question_order', $viewParams);
	}

	public function actionAddTypeChooser(ParameterBag $params)
	{
		/** @var Form $form */
		$form = $this->em()->findOne('Snog\Forms:Form', ['posid', '=', $params['posid']]);

		$questionRepo = $this->getQuestionRepo();
		$typeData = $questionRepo->getQuestionTypeData();

		$conditional = $this->filter('conditional', 'bool');

		if ($this->isPost())
		{
			$type = $this->filter('type', 'str');
			if (!isset($typeData[$type]))
			{
				return $this->notFound();
			}

			if ($conditional)
			{
				return $this->redirect($this->buildLink('form-editquestions/addcon', $form, ['type' => $type]));
			}
			return $this->redirect($this->buildLink('form-editquestions/add', $form, ['type' => $type]));
		}

		$viewParams = [
			'form' => $form,
			'typeData' => $typeData,
			'conditional' => $conditional,
		];
		return $this->view(
			'Snog:Forms\Question',
			'snog_forms_question_add_type_chooser',
			$viewParams
		);
	}

	/**
	 * @param ParameterBag $params
	 * @return \XF\Mvc\Reply\AbstractReply|\XF\Mvc\Reply\Redirect|\XF\Mvc\Reply\View
	 * @throws \XF\Mvc\Reply\Exception
	 */
	public function actionChangeType(ParameterBag $params)
	{
		$question = $this->assertQuestionExists($params->questionid);

		$questionRepo = $this->getQuestionRepo();
		$typeData = $questionRepo->getQuestionTypeData();

		if ($this->isPost())
		{
			$type = $this->filter('type', 'str');
			if (!isset($typeData[$type]))
			{
				return $this->notFound();
			}

			$question->type = $type;
			$question->saveIfChanged();

			if ($question->conditional)
			{
				return $this->redirect($this->buildLink('form-questions/conedit', $question));
			}
			return $this->redirect($this->buildLink('form-questions/edit', $question));
		}

		$viewParams = [
			'question' => $question,
			'typeData' => $typeData
		];
		return $this->view(
			'Snog:Forms\Question',
			'snog_forms_question_change_type',
			$viewParams
		);
	}

	public function actionAdd(ParameterBag $params)
	{
		$type = $this->filter('type', 'str');
		if (!$type)
		{
			return $this->redirect($this->buildLink('form-editquestions/add-type-chooser'));
		}

		/** @var Form $editForm */
		$editForm = $this->em()->findOne('Snog\Forms:Form', ['posid', '=', $params->posid]);

		$questionRepo = $this->getQuestionRepo();

		/** @var Question[] $questions */
		$questions = $questionRepo->createQuestionTree($questionRepo->getQuestionList($params->posid));

		/** @var Question[] $checkQuestions */
		$checkQuestions = $questionRepo->getQuestionList($params->posid);
		$nextQuestion = count($questions) + 1;

		$typesPresent = [];
		foreach ($checkQuestions as $existing)
		{
			// CHECK IF THERE IS ALREADY A FILE UPLOAD QUESTION FOR THIS FORM
			if ($existing->type == 'file_upload')
			{
				$typesPresent[] = 'file_upload';
			}

			// CHECK IF THERE IS ALREADY A PREFIX QUESTION FOR THIS FORM
			if ($existing->type == 'thread_prefix')
			{
				$typesPresent[] = 'thread_prefix';
			}
		}

		/** @var Question $question */
		$question = $this->em()->create('Snog\Forms:Question');
		if ($editForm)
		{
			$question->posid = $editForm->posid;
		}

		$question->type = $type;

		return $this->questionAddEdit($question, $editForm, $nextQuestion, $typesPresent);
	}

	public function actionAddCon(ParameterBag $params)
	{
		/** @var Form $editForm */
		$editForm = $this->em()->findOne('Snog\Forms:Form', ['posid', '=', $params['posid']]);

		$type = $this->filter('type', 'str');
		if (!$type)
		{
			return $this->redirect($this->buildLink(
				'form-editquestions/add-type-chooser', $editForm, ['conditional' => true]
			));
		}

		$questionRepo = $this->getQuestionRepo();

		/** @var Question[] $questions */
		$questions = $questionRepo->createQuestionTree($questionRepo->getQuestionList($params['posid']));

		/** @var Question[] $checkQuestions */
		$checkQuestions = $questionRepo->getQuestionList($params['posid']);
		$nextQuestion = count($questions) + 1;
		$typesPresent = [];
		$conditionals = [];

		foreach ($checkQuestions as $existing)
		{
			// CHECK IF THERE IS ALREADY A FILE UPLOAD QUESTION FOR THIS FORM
			if ($existing->type == 'file_upload')
			{
				$typesPresent[] = 'file_upload';
			}

			// CHECK IF THERE IS ALREADY A PREFIX QUESTION FOR THIS FORM
			if ($existing->type == 'thread_prefix')
			{
				$typesPresent[] = 'thread_prefix';
			}

			// GET QUESTIONS THAT CAN TRIGGER CONDITIONALS
			if ($existing->canTriggerConditionals())
			{
				if ($existing->type == 'yes_no')
				{
					$answers = [
						'1' => \XF::phrase('yes'),
						'2' => \XF::phrase('no')
					];
				}
				else
				{
					$answers = preg_split('/\r?\n/', $existing->expected);
				}

				$conditionals[] = [
					'questionId' => $existing->questionid,
					'text' => $existing->text,
					'type' => $existing->type,
					'answers' => $answers
				];
			}
		}

		if (empty($conditionals))
		{
			return $this->error(\XF::phrase('snog_forms_error_none_defined'));
		}

		/** @var Question $question */
		$question = $this->em()->create('Snog\Forms:Question');
		$question->posid = $editForm->posid;
		$question->conditional = PHP_INT_MAX; // Trick to pass conditions in Question::canAddQuestionType()
		$question->type = $type;

		return $this->conQuestionAddEdit($question, $editForm, $nextQuestion, $typesPresent, $conditionals);
	}

	public function questionAddEdit($question, $editForm, $nextQuestion = 0, $typesPresent = [])
	{
		/** @var \XF\Repository\Node $nodeRepo */
		$nodeRepo = $this->repository('XF:Node');

		$questionRepo = $this->getQuestionRepo();
		$questions = $questionRepo->getQuestionList($editForm);

		$viewParams = [
			'question' => $question,
			'questions' => $questions,
			'form' => $editForm,
			'nextquestion' => $nextQuestion,
			'typesPresent' => $typesPresent,
			'nodeTree' => $nodeRepo->createNodeTree($nodeRepo->getFullNodeList()),
		];

		return $this->view('Snog:Forms\Question', 'snog_forms_question_edit', $viewParams);
	}

	public function conQuestionAddEdit($question, $editForm, $nextQuestion = 0, $typesPresent = [], $conditionals = null)
	{
		$questionRepo = $this->getQuestionRepo();
		$questions = $questionRepo->getQuestionList($editForm);

		$viewParams = [
			'question' => $question,
			'questions' => $questions,
			'form' => $editForm,
			'nextquestion' => $nextQuestion,
			'typesPresent' => $typesPresent,
			'conditionals' => $conditionals
		];

		return $this->view('Snog:Forms\Question', 'snog_forms_con_question_edit', $viewParams);
	}

	/**
	 * @param ParameterBag $params
	 * @return \XF\Mvc\Reply\View
	 * @throws \XF\Mvc\Reply\Exception
	 */
	public function actionEdit(ParameterBag $params)
	{
		$question = $this->assertQuestionExists($params['questionid']);
		$questionRepo = $this->getQuestionRepo();

		/** @var Question[] $questions */
		$questions = $questionRepo->getQuestionList($question['posid']);
		$typesPresent = [];

		foreach ($questions as $existing)
		{
			// CHECK IF THERE IS ALREADY A FILE UPLOAD QUESTION FOR THIS FORM
			if ($existing->type == 'file_upload' && $existing->questionid !== $question->questionid)
			{
				$typesPresent[] = 'file_upload';
			}

			// CHECK IF THERE IS ALREADY A PREFIX QUESTION FOR THIS FORM
			if ($existing->type == 'thread_prefix' && $existing->questionid !== $question->questionid)
			{
				$typesPresent[] = 'thread_prefix';
			}
		}

		/** @var Form $editForm */
		$editForm = $this->em()->findOne('Snog\Forms:Form', ['posid', '=', $question['posid']]);
		return $this->questionAddEdit($question, $editForm, 0, $typesPresent);
	}

	/**
	 * @param ParameterBag $params
	 * @return \XF\Mvc\Reply\View
	 * @throws \XF\Mvc\Reply\Exception
	 */
	public function actionConEdit(ParameterBag $params)
	{
		$question = $this->assertQuestionExists($params['questionid']);
		$questionRepo = $this->getQuestionRepo();

		/** @var Question[] $questions */
		$questions = $questionRepo->getQuestionList($question['posid']);
		$typesPresent = [];
		$conditionals = [];

		foreach ($questions as $existing)
		{
			// CHECK IF THERE IS ALREADY A FILE UPLOAD QUESTION FOR THIS FORM
			if ($existing->type == 'file_upload' && $existing->questionid !== $question->questionid)
			{
				$typesPresent[] = 'file_upload';
			}

			// CHECK IF THERE IS ALREADY A PREFIX QUESTION FOR THIS FORM
			if ($existing->type == 'thread_prefix' && $existing->questionid !== $question->questionid)
			{
				$typesPresent[] = 'thread_prefix';
			}

			// QUESTIONS THAT CAN BE USED AS CONDITIONAL TRIGGERS
			if ($existing->canTriggerConditionals())
			{
				if ($existing->type == 'yes_no')
				{
					$answers = [
						'1' => \XF::phrase('yes'),
						'2' => \XF::phrase('no')
					];
				}
				else
				{
					$answers = preg_split('/\r?\n/', $existing->expected);
				}

				$conditionals[] = [
					'questionId' => $existing->questionid,
					'text' => $existing->text,
					'type' => $existing->type,
					'answers' => $answers
				];
			}
		}

		$editForm = $this->em()->findOne('Snog\Forms:Form', ['posid', '=', $question['posid']]);
		return $this->conQuestionAddEdit($question, $editForm, 0, $typesPresent, $conditionals);
	}

	/**
	 * @param ParameterBag $params
	 * @return \XF\Mvc\Reply\Redirect
	 * @throws \XF\Mvc\Reply\Exception
	 * @throws \XF\PrintableException
	 */
	public function actionSave(ParameterBag $params)
	{
		if ($params->questionid)
		{
			/** @var Question $modifiedQuestion */
			$modifiedQuestion = $this->assertQuestionExists($params->questionid);
		}
		else
		{
			/** @var Question $modifiedQuestion */
			$modifiedQuestion = $this->em()->create('Snog\Forms:Question');
		}

		$this->questionSaveProcess($modifiedQuestion)->run();

		// ADD CONDITIONAL TO MASTER QUESTION CONDITIONAL LIST
		if ($modifiedQuestion['conditional'])
		{
			/** @var Question $masterQuestion */
			$masterQuestion = $this->em()->findOne('Snog\Forms:Question', ['questionid', '=', $modifiedQuestion['conditional']]);
			$existingConditionals = $masterQuestion->hasconditional;
			if (!$existingConditionals || !in_array($modifiedQuestion['questionid'], $existingConditionals))
			{
				$existingConditionals[] = $modifiedQuestion->questionid;
			}
			$masterQuestion->hasconditional = $existingConditionals;
			$masterQuestion->save();
		}

		/** @var Form $editForm */
		$editForm = $this->finder('Snog\Forms:Form')->where('posid', $modifiedQuestion['posid'])->fetchOne();

		if ($editForm)
		{
			return $this->redirect($this->buildLink('form-editquestions/formquestions', $editForm));
		}
		else
		{
			return $this->redirect($this->buildLink('form-questions'));
		}
	}

	/**
	 * @param Question $question
	 * @return FormAction
	 * @throws \XF\Mvc\Reply\Exception
	 * @throws \XF\PrintableException
	 */
	protected function questionSaveProcess(Question $question)
	{
		$formAction = $this->formAction();
		$input = $this->filter([
			'text' => 'str',
			'description' => 'str',
			'type' => 'str',
			'error' => 'str',
			'expected' => 'str',
			'display' => 'uint',
			'regex' => 'str',
			'regexerror' => 'str',
			'showquestion' => 'bool',
			'showunanswered' => 'bool',
			'questionpos' => 'uint',
			'posid' => 'uint',
			'inline' => 'uint',
			'format' => 'str',
			'conditional' => 'uint',
			'conanswer' => 'str',
			'checklimit' => 'uint',
			'checkmin' => 'uint',
			'checkerror' => 'str',
			'type_data' => 'array',
			'readonly' => 'bool'
		]);

		// NO ERROR FOR QUESTION - CHECK IF QUESTION IS USED IN FORM TITLE
		if (!$input['error'] && $question->posid)
		{
			$form = $question->Form;

			preg_match_all('/({A\d+})/', $form->subject, $titleAnswers);
			$questionIds = [];

			if (!empty($titleAnswers[1]))
			{
				foreach ($titleAnswers[1] as $titleAnswer)
				{
					$questionNumber = str_replace('{A', '', $titleAnswer);
					$questionNumber = str_replace('}', '', $questionNumber);
					$questionIds[] = $questionNumber;
				}
			}

			// IT'S IN THE FORM TITLE - THROW AN EXCEPTION
			if (in_array($question->display, $questionIds))
			{
				throw $this->exception($this->notFound(\XF::phrase('snog_forms_error_question_used')));
			}
		}

		// ASSIGN PLACEHOLDER, EXPECTED ANSWERS, DEFAULT ANSWER AND REGEX VALUES
		$placeholders = $this->filter([
			'singleline' => 'str',
			'multiline' => 'str',
			'wysiwyg' => 'str',
		]);

		$expecteds = $this->filter([
			'expectedmulti' => 'str',
			'expectedmultiall' => 'str',
			'expectedagree' => 'str',
			'expectedsingle' => 'str',
			'expectedradio' => 'str',
			'expectedspin' => 'str',
		]);

		$defaults = $this->filter([
			'defsingleline' => 'str',
			'defmultiline' => 'str',
			'defyesno' => 'str',
			'defmulticheck' => 'str',
			'defmulticheckall' => 'str',
			'defsingledrop' => 'str',
			'defsingleforum' => 'str',
			'defradio' => 'str',
			'defspin' => 'str',
		]);

		$regs = $this->filter([
			'regexsingle' => 'str',
			'regexerrorsingle' => 'str',
			'regexmulti' => 'str',
			'regexerrormulti' => 'str',
			'regexspin' => 'str',
			'regexerrorspin' => 'str',
		]);

		switch ($input['type'])
		{
			case 'text':
				$input['placeholder'] = $placeholders['singleline'];
				$input['defanswer'] = $defaults['defsingleline'];
				$input['regex'] = $regs['regexsingle'];
				$input['regexerror'] = $regs['regexerrorsingle'];
				break;

			case 'multiline_text':
				$input['placeholder'] = $placeholders['multiline'];
				$input['defanswer'] = $defaults['defmultiline'];
				$input['regex'] = $regs['regexmulti'];
				$input['regexerror'] = $regs['regexerrormulti'];
				break;

			case 'yes_no':
				$input['defanswer'] = $defaults['defyesno'];
				break;

			case 'radio':
				$input['expected'] = $expecteds['expectedradio'];
				$input['defanswer'] = $defaults['defradio'];
				break;

			case 'checkboxes':
				$input['expected'] = $expecteds['expectedmulti'];
				$input['defanswer'] = $defaults['defmulticheck'];
				break;

			case 'select':
				$input['expected'] = $expecteds['expectedsingle'];
				$input['defanswer'] = $defaults['defsingledrop'];
				break;

			case 'multi_select':
				$input['expected'] = $expecteds['expectedmultiall'];
				$input['defanswer'] = $defaults['defmulticheckall'];
				break;

			case 'checkbox':
				$input['expected'] = $expecteds['expectedagree'];
				break;

			case 'forum_select':
				$input['defanswer'] = $defaults['defsingleforum'];
				break;

			case 'wysiwyg':
				$input['placeholder'] = $placeholders['wysiwyg'];
				break;

			case 'spinbox':
				$input['expected'] = $expecteds['expectedspin'];
				$input['defanswer'] = $defaults['defspin'];
				$input['regex'] = $regs['regexspin'];
				$input['regexerror'] = $regs['regexerrorspin'];
				break;

			case 'billable_row':
				$input['expected'] = $expecteds['expectedspin'];
				$input['defanswer'] = $defaults['defspin'];
		}

		$originalConditional = $this->filter('originalConditional', 'uint');
		$isConditional = $this->filter('isconditional', 'uint');

		// HEY DUMMY! YOU FORGOT TO SELECT A QUESTION TO TRIGGER THIS CONDITIONAL QUESTION
		if ($isConditional && !$input['conditional'])
		{
			$formAction->logError(\XF::phrase('snog_forms_error_conditional_question'));
		}

		// HEY DUMMY! YOU FORGOT TO SELECT AN ANSWER TO TRIGGER THIS CONDITIONAL QUESTION
		if ($isConditional && !$input['conanswer'])
		{
			$formAction->logError(\XF::phrase('snog_forms_error_conditional_answer'));
		}

		// CONDITIONAL CHANGED - REMOVE CONDITIONAL FROM ORIGINAL MASTER QUESTION CONDITIONAL LIST
		if ($originalConditional && $input['conditional'] !== $originalConditional)
		{
			/** @var Question $oldMasterQuestion */
			$oldMasterQuestion = $this->em()->findOne('Snog\Forms:Question', ['questionid', '=', $originalConditional]);
			if ($oldMasterQuestion)
			{
				$existingConditionals = $oldMasterQuestion->hasconditional;
				if (($key = array_search($question->questionid, $existingConditionals)) !== false)
				{
					unset($existingConditionals[$key]);
				}
				$oldMasterQuestion->hasconditional = $existingConditionals;
				$oldMasterQuestion->save();
			}
		}

		// CLEAN EXPECTED ANSWERS FOR MULTIPLE CHOICE QUESTIONS OF EMPTY LINES IF PRESENT
		if (in_array($input['type'], [
			'radio',
			'checkboxes',
			'select',
			'multi_select'
		]))
		{
			$expectedArray = preg_replace("/(^[\r\n]*|[\r\n]+)[\s\t]*[\r\n]+/", "\n", $input['expected']);
			$input['expected'] = $expectedArray;
		}

		$formAction->basicEntitySave($question, $input);
		return $formAction;
	}

	/**
	 * @param ParameterBag $params
	 * @return \XF\Mvc\Reply\Error|\XF\Mvc\Reply\Redirect|\XF\Mvc\Reply\View
	 * @throws \XF\Mvc\Reply\Exception
	 * @throws \XF\PrintableException
	 */
	public function actionDelete(ParameterBag $params)
	{
		$question = $this->assertQuestionExists($params->questionid);

		if ($this->isPost())
		{
			$question->preDelete();
			$errors = $question->getErrors();
			if ($errors)
			{
				return $this->error($errors);
			}
			$question->delete();

			/** @var Form $editForm */
			$editForm = $this->em()->findOne('Snog\Forms:Form', ['posid', '=', $question['posid']]);

			if ($editForm)
			{
				return $this->redirect($this->buildLink('form-editquestions/formquestions', $editForm));
			}

			return $this->redirect($this->buildLink('form-questions'));
		}

		$viewParams = ['question' => $question];
		return $this->view('Snog:Forms\Question', 'snog_forms_confirm', $viewParams);
	}

	/**
	 * @param $id
	 * @param null $with
	 * @return Form|\XF\Mvc\Entity\Entity
	 * @throws \XF\Mvc\Reply\Exception
	 * @noinspection PhpReturnDocTypeMismatchInspection
	 */
	protected function assertFormExists($id, $with = null)
	{
		return $this->assertRecordExists('Snog\Forms:Form', $id, $with, 'snog_forms_form_not_found');
	}

	/**
	 * @param $id
	 * @param null $with
	 * @return Question|\XF\Mvc\Entity\Entity
	 * @throws \XF\Mvc\Reply\Exception
	 * @noinspection PhpReturnDocTypeMismatchInspection
	 */
	protected function assertQuestionExists($id, $with = null)
	{
		return $this->assertRecordExists('Snog\Forms:Question', $id, $with, 'snog_forms_question_not_found');
	}

	/**
	 * @return \Snog\Forms\Repository\Question|\XF\Mvc\Entity\Repository
	 * @noinspection PhpReturnDocTypeMismatchInspection
	 */
	protected function getQuestionRepo()
	{
		return $this->repository('Snog\Forms:Question');
	}
}