<?php

namespace Snog\Forms\Admin\Controller;

use Snog\Forms\Globals;
use XF\Admin\Controller\AbstractController;
use XF\Mvc\ParameterBag;
use XF\Payment\CallbackState;

class SubmitPurchases extends AbstractController
{
	protected function preDispatchController($action, ParameterBag $params)
	{
		$this->assertAdminPermission('viewLogs');
	}

	public function actionIndex()
	{
		$page = $this->filterPage();
		$perPage = $this->filter('limit', 'int', 20);
		if ($perPage > 1000)
		{
			$perPage = 1000;
		}

		/** @var \Snog\Forms\Repository\SubmitPurchase $purchaseRepo */
		$purchaseRepo = $this->repository('Snog\Forms:SubmitPurchase');
		$entryFinder = $purchaseRepo->findPurchasesForList()
			->with('User')
			->with('Form');

		$this->applyFilters($entryFinder);
		$entryFinder->limitByPage($page, $perPage);

		$filters = $this->getFilterInput(true);
		$entries = $entryFinder->fetch();

		if ($this->isPost() && $filters)
		{
			return $this->redirect($this->buildLink('form/purchases', null, $filters));
		}

		$viewParams = [
			'entries' => $entries,
			'filters' => $filters,
			'page' => $page,
			'perPage' => $perPage,
			'total' => $entryFinder->total()
		];
		return $this->view(
			'Snog\Forms:SubmitPurchases\Listing',
			'snog_forms_submit_purchase_list',
			$viewParams
		);
	}

	public function actionRefineSearch()
	{
		return $this->view('Snog\Forms:SubmitPurchases\RefineSearch',
			'snog_forms_submit_purchase_refine_search', [
				'datePresets' => \XF::language()->getDatePresets(),
				'conditions' => $this->getFilterInput(),
			]);
	}

	protected function applyFilters(\XF\Mvc\Entity\Finder $finder)
	{
		$filters = $this->getFilterInput();

		if ($filters['username'])
		{
			$finder->where('User.username', 'LIKE', $finder->escapeLike($filters['username'], '%?%'));
		}
		if ($filters['title'])
		{
			$finder->where('Form.position', 'LIKE', $finder->escapeLike($filters['title'], '%?%'));
		}
		if ($filters['form_id'])
		{
			$finder->where('form_id', '=', $filters['form_id']);
		}

		if ($filters['purchase_states'] && !in_array('', $filters['purchase_states']))
		{
			$finder->where('purchase_state', $filters['purchase_states']);
		}

		if ($filters['start'])
		{
			$finder->where('purchase_date', '>', $filters['start']);
		}
		if ($filters['end'])
		{
			$finder->where('purchase_date', '<', $filters['end']);
		}
	}

	protected function getFilterInput($removeEmpty = false): array
	{
		$input = $this->filter([
			'username' => 'str',
			'title' => 'str',
			'form_id' => 'uint',
			'purchase_states' => 'array-str',
			'start' => 'datetime',
			'end' => 'datetime',
		]);

		if ($removeEmpty)
		{
			$input = array_filter($input);
		}

		return $input;
	}

	public function actionDelete(ParameterBag $params)
	{
		$submitPurchase = $this->assertSubmitPurchaseExists($params->purchase_id);

		/** @var \XF\ControllerPlugin\Delete $plugin */
		$plugin = $this->plugin('XF:Delete');
		return $plugin->actionDelete(
			$submitPurchase,
			$this->buildLink('form/purchases/delete', $submitPurchase),
			$this->buildLink('form/purchases', $submitPurchase),
			$this->buildLink('form/purchases'),
			$submitPurchase->Form ? $submitPurchase->Form->position : ''
		);
	}

	public function actionResendPaymentReceipt(ParameterBag $params)
	{
		$submitPurchase = $this->assertSubmitPurchaseExists($params->purchase_id, [
			'PaymentProviderLog.Provider',
			'PaymentProviderLog.PurchaseRequest',
			'PaymentProviderLog.PurchaseRequest.PaymentProfile',
			'PaymentProviderLog.PurchaseRequest.Purchasable',
			'PaymentProviderLog.PurchaseRequest.User',
		]);

		/** @var \XF\Entity\PaymentProviderLog $providerLog */
		$providerLog = $submitPurchase->PaymentProviderLog;
		$purchaseRequest = $providerLog->PurchaseRequest;
		if ($purchaseRequest)
		{
			$purchasable = $providerLog->PurchaseRequest->Purchasable;
		}
		else
		{
			$purchasable = null;
		}

		if (!$purchasable || !$purchasable->handler)
		{
			return $this->notFound();
		}

		$purchasableHandler = $purchasable->handler;

		$state = new CallbackState;
		$state->purchaseRequest = $purchaseRequest;

		if ($providerLog->log_type != 'payment')
		{
			return $this->error(\XF::phrase('snog_forms_failed_to_resend_payment_receipt'));
		}

		$state->paymentResult = CallbackState::PAYMENT_RECEIVED;

		$purchasableHandler->sendPaymentReceipt($state);

		return $this->message(\XF::phrase('snog_forms_payment_receipt_successfully_resend'));
	}

	/**
	 * @param      $id
	 * @param null $with
	 * @param null $phraseKey
	 *
	 * @return \XF\Mvc\Entity\Entity|\Snog\Forms\Entity\SubmitPurchase
	 * @throws \XF\Mvc\Reply\Exception
	 */
	protected function assertSubmitPurchaseExists($id, $with = null, $phraseKey = null)
	{
		return $this->assertRecordExists('Snog\Forms:SubmitPurchase', $id, $with, $phraseKey);
	}
}