<?php

namespace Snog\Forms\Entity;

use XF\Mvc\Entity\Entity;
use XF\Mvc\Entity\Structure;

/**
 * COLUMNS
 * @property int $questionid
 * @property int $posid
 * @property string $text
 * @property string $description
 * @property string $type
 * @property string $error
 * @property string $expected
 * @property int $display
 * @property int $display_parent
 * @property string $regex
 * @property string $regexerror
 * @property string $defanswer
 * @property bool $readonly
 * @property int $questionpos
 * @property int $showquestion
 * @property int $showunanswered
 * @property int $inline
 * @property string $format
 * @property array $hasconditional
 * @property int $conditional
 * @property string $conanswer
 * @property string $placeholder
 * @property int $checklimit
 * @property int $checkmin
 * @property string $checkerror
 * @property array $type_data
 *
 * RELATIONS
 * @property Form $Form
 * @property Question $ConditionalQuestion
 */
class Question extends Entity implements ExportableInterface
{
	/**
	 * @Deprecated
	 */
	const TYPE_TEXT = 'text';
	const TYPE_MULTILINE_TEXT = 'multiline_text';
	const TYPE_YES_NO = 'yes_no';
	const TYPE_RADIO = 'radio';
	const TYPE_CHECKBOXES = 'checkboxes';
	const TYPE_HEADER_PHRASE = 'header_phrase';
	const TYPE_SELECT = 'select';
	const TYPE_MULTI_SELECT = 'multi_select';
	const TYPE_CHECKBOX = 'checkbox';
	const TYPE_FILE_UPLOAD = 'file_upload';
	const TYPE_DATE_INPUT = 'date_input';
	const TYPE_FORUM_SELECT = 'forum_select';
	const TYPE_WYSIWYG = 'wysiwyg';
	const TYPE_SPINBOX = 'spinbox';
	const TYPE_THREAD_PREFIX = 'thread_prefix';
	const TYPE_DATETIME_INPUT = 'datetime_input';
	const TYPE_TIME_INPUT = 'time_input';

	public function getTypeTitle()
	{
		return \XF::phrase('snog_forms_question_type_def.' . $this->type);
	}

	public function getTypeDescription()
	{
		return \XF::phrase('snog_forms_question_type_def_desc.' . $this->type);
	}

	public function isQuestionTypeAllowed($questionType)
	{
		if ($questionType == 'file_upload')
		{
			return $this->db()->fetchRow('
				SELECT COUNT(*) FROM xf_snog_forms_questions
				WHERE posid = ? AND type = ? AND questionid != ? AND conditional = 0
			', [$this->posid, $questionType, $this->questionid]);
		}

		return true;
	}

	public function getDataForCopy()
	{
		$output = [];
		foreach ($this->_structure->columns as $key => $column)
		{
			$value = $this->getValue($key);
			if ($value !== null && empty($column['nullable']))
			{
				$output[$key] = $value;
			}
		}
		unset($output['questionid']);

		return $output;
	}

	public function getQuestionFormat($context = 'message')
	{
		$questionFormat = $this->format;

		if ($this->hasAnswerMessage())
		{
			// Question position in post if not on the same line
			if (!$this->getOption('is_first'))
			{
				switch ($this->questionpos)
				{
					case 1:
						$questionFormat = "\r\n\r\n" . $this->format;
						break;
					case 2:
						$questionFormat = "\r\n" . $this->format;
				}
			}
		}

		return $questionFormat;
	}

	public function getFormattedReportMessage($answer, $context = 'message')
	{
		$questionFormat = $this->getQuestionFormat($context);

		// Not a header or thread prefix, add to message
		if ($this->hasAnswerMessage())
		{
			// Display question if selected
			$messageQuestion = $this->getShownQuestion($context);

			$messageAnswer = $this->getWrappedAnswerMessage(
				$this->getFormattedAnswer($answer, $context)
			);

			$questionResult = str_ireplace('{question}', $messageQuestion, $questionFormat);
			return str_ireplace('{answer}', $messageAnswer, $questionResult);
		}

		return '';
	}

	public function getDefaultAnswer(\XF\Entity\User $user = null)
	{
		$defAnswer = $this->defanswer;
		if ($defAnswer == '')
		{
			return $defAnswer;
		}

		if (!$user)
		{
			$user = \XF::visitor();
		}

		$defAnswer = str_replace('{username}', $user->username, $defAnswer);
		$defAnswer = str_replace('{userid}', $user->user_id, $defAnswer);
		$defAnswer = str_replace('{email}', $user->email, $defAnswer);

		if ($user->Profile)
		{
			$defAnswer = str_replace('{location}', $user->Profile->location, $defAnswer);
		}

		$customFields = $user->Profile->custom_fields;
		if ($customFields)
		{
			if ($user->user_id)
			{
				foreach ($customFields as $key => $field)
				{
					if ($field)
					{
						if (is_array($field))
						{
							$defAnswer = str_replace('{custom.' . $key . '}', implode(',', $field), $defAnswer);
						}
						else
						{
							$defAnswer = str_replace('{custom.' . $key . '}', $field, $defAnswer);
						}
					}
					else
					{
						$defAnswer = str_replace('{custom.' . $key . '}', '', $defAnswer);
					}
				}
			}
			else
			{
				/** @var \XF\CustomField\DefinitionSet $definitionSet */
				$definitionSet = $this->app()->container('customFields.users');
				$fieldDefinitions = $definitionSet->getFieldDefinitions();

				foreach ($fieldDefinitions as $key => $field)
				{
					$defAnswer = str_replace('{custom.' . $key . '}', $defAnswer, '');
				}
			}
		}

		return $defAnswer;
	}

	public function isValidConditionalAnswer($conditionAnswer, $answer)
	{
		return $conditionAnswer == $answer
			|| (is_array($answer) && in_array($conditionAnswer, $answer))
			|| in_array($this->type, ['date_input', 'datetime_input', 'time_input']);
	}

	public function isUploadNeeded()
	{
		return $this->type == 'file_upload';
	}

	public function hasUrl()
	{
		return (bool) stristr($this->text, '[URL');
	}

	public function hasExpectedAnswer()
	{
		return $this->expected && in_array($this->type, ['radio', 'checkboxes', 'select', 'multi_select']);
	}

	/**
	 * @return array
	 */
	public function getExpectedAnswers()
	{
		if ($this->hasExpectedAnswer())
		{
			return preg_split('/\r?\n/', $this->expected) ?: [];
		}

		return [];
	}

	public function isAnswerStored()
	{
		return !in_array($this->type, ['header_phrase', 'file_upload']);
	}

	public function isUnanswered()
	{
		if (in_array($this->type, ['header_phrase', 'thread_prefix']))
		{
			return true;
		}
		elseif ($this->type == 'checkbox')
		{
			if (empty($this->expected))
			{
				return true;
			}
		}

		return false;
	}

	public function hasAnswerMessage()
	{
		return !in_array($this->type, ['header_phrase', 'thread_prefix']);
	}

	public function getFormattedAnswer($answer, $context = 'message')
	{
		if ($answer === null
			|| ($this->type == 'select' && $answer == 0)
			|| ($this->type == 'date_input' && $answer == '')
			|| ($this->type == 'datetime_input' && $answer == '')
			|| ($this->type == 'time_input' && $answer == '')
			|| ($this->type == 'forum_select' && $answer == 0)
			|| ($this->type == 'billable_row_total' && $answer == null)
		)
		{
			// USER DID NOT ANSWER QUESTION
			if ($this->type == 'checkbox')
			{
				if (!empty($this->expected))
				{
					return \XF::phrase('snog_forms_no_answer');
				}
			}
			if ($this->type == 'billable_row_total')
			{
				$form = $this->Form;
				$answers = $form->getOption('answers');
				if (!empty($this->type_data['question_ids']))
				{
					$total = 0;
					foreach ($this->type_data['question_ids'] as $questionId)
					{
						$question = $form->Questions[$questionId] ?? null;
						if ($question && !empty($question['type_data']['cost']))
						{
							$value = $answers[$questionId] ?? 0;
							$total += $value * $question['type_data']['cost'];
						}
					}

					$str1 = "{currency}{total}";
					return strtr($str1, [
						'{total}' => $total,
						'{currency}' => $this->type_data['currency'] ?? '',
					]);
				}
			}
			else
			{
				return \XF::phrase('snog_forms_no_answer');
			}
		}
		else
		{
			if ($this->type == 'date_input')
			{
				return $this->getDateAnswer($answer, $context);
			}
			elseif ($this->type == 'datetime_input')
			{
				return $this->getDateTimeAnswer($answer, $context);
			}
			elseif ($this->type == 'time_input')
			{
				return $this->getTimeAnswer($answer, $context);
			}
			elseif ($this->type == 'checkbox')
			{
				if (!empty($this->expected))
				{
					return $answer;
				}
			}
			elseif ($this->type == 'yes_no')
			{
				return $this->getYesNoAnswer($answer, $context);
			}
			elseif (in_array($this->type, ['radio', 'select']) && $this->expected)
			{
				return $this->getRadioSelectAnswer($answer, $context);
			}
			elseif ($this->type == 'forum_select')
			{
				return $this->getForumSelectAnswer($answer, $context);
			}
			elseif (is_array($answer))
			{
				return $this->getMultipleChoicesAnswer($answer, $context);
			}
			elseif ($this->type == 'billable_row')
			{
				$str = "{answer} x {currency}{cost}";

				return strtr($str, [
					'{answer}' => $answer,
					'{cost}' => $this->type_data['cost'] ?? 0,
					'{currency}' => $this->type_data['currency'] ?? '',
				]);

			}
			else
			{
				return $answer;
			}
		}

		return strval($answer);
	}

	public function canUsedForReportTitle()
	{
		return in_array($this->type, [
			'text',
			'yes_no',
			'radio',
			'select',
			'date_input',
			'datetime_input',
			'time_input',
			'forum_select'
		]);
	}

	public function getTitleAnswer($answer)
	{
		if ($this->canUsedForReportTitle())
		{
			if ($this->type == 'date_input')
			{
				return $this->getDateAnswer($answer, 'title');
			}
			elseif ($this->type == 'datetime_input')
			{
				return $this->getDateTimeAnswer($answer, 'title');
			}
			elseif ($this->type == 'time_input')
			{
				return $this->getDateTimeAnswer($answer, 'title');
			}
			elseif ($this->type == 'yes_no')
			{
				return $this->getYesNoAnswer($answer, 'title');
			}
			elseif ($this->expected && in_array($this->type, ['radio', 'select']))
			{
				return $this->getRadioSelectAnswer($answer, 'title');
			}
			elseif ($this->type == 'forum_select')
			{
				return $this->getForumSelectAnswer($answer, 'title');
			}
			elseif ($this->type == 'text')
			{
				return $answer;
			}
		}

		return '';
	}

	public function getMultipleChoicesAnswer($answers, $context = 'message')
	{
		if (is_array($answers))
		{
			$expectedArray = preg_split('/\r?\n/', $this->expected);
			$answerArray = [];

			foreach ($answers as $answer)
			{
				if ($answer !== 'all')
				{
					$answerId = intval($answer);
					$answer = $expectedArray[$answerId - 1] ?? null;
					if ($answer)
					{
						$answerArray[] = trim($answer);
					}
				}
			}

			return implode(', ', $answerArray);
		}

		return '';
	}

	public function getForumSelectAnswer($answer, $context = 'message')
	{
		/** @var \XF\Entity\Forum $forum */
		$forum = $this->em()->findCached('XF:Node', $answer);
		if (!$forum)
		{
			$forum = $this->em()->find('XF:Node', $answer);
		}

		if ($forum)
		{
			return $forum->title;
		}

		return '';
	}

	public function getRadioSelectAnswer($answer, $context = 'message')
	{
		$answerArray = preg_split('/\r?\n/', $this->expected);
		return isset($answerArray[$answer - 1]) ? trim($answerArray[$answer - 1]) : '';
	}

	public function getYesNoAnswer($answer, $context = 'message')
	{
		return $answer == 1 ? \XF::phrase('yes') : \XF::phrase('no');
	}

	public function getDateAnswer($answer, $context = 'message')
	{
		$timestamp = \XF::$time;
		if (!empty($answer))
		{
			$timestamp = strtotime($answer);
		}

		$newDate = date('Y-m-d', $timestamp);

		if (isset($this->type_data['format_user_timezone']))
		{
			$visitor = \XF::visitor();
			$newDate = "$newDate 12:00 $visitor->timezone";
		}
		else
		{
			$newDate = "$newDate 12:00";
		}

		$newStart = strtotime($newDate);
		return $this->app()->language()->date($newStart);
	}

	public function getDateTimeAnswer($answer, $context = 'message')
	{
		$visitor = \XF::visitor();

		if (is_array($answer))
		{
			$answer = implode(' ', $answer);
		}

		$timestamp = \XF::$time;
		if (!empty($answer))
		{
			if (isset($this->type_data['format_user_timezone']))
			{
				$timestamp = strtotime("$answer $visitor->timezone");
			}
			else
			{
				$timestamp = strtotime($answer);
			}
		}

		return $this->app()->language()->dateTime($timestamp);
	}

	public function getTimeAnswer($answer, $context = 'message')
	{
		$visitor = \XF::visitor();

		if (isset($this->type_data['format_user_timezone']))
		{
			$convertDate = strtotime("1970-01-01 $answer $visitor->timezone");
		}
		else
		{
			$convertDate = strtotime("1970-01-01 $answer");
		}

		return $this->app()->language()->time($convertDate);
	}

	public function getShownQuestion($context = 'message')
	{
		$messageQuestion = '';

		if ($this->showquestion)
		{
			// AGREEMENT CHECK BOX
			if ($this->type == 'checkbox')
			{
				if ($this->expected > '')
				{
					$messageQuestion .= $this->expected;
				}
			}
			else
			{
				// REGULAR QUESTION
				$messageQuestion .= $this->text;
			}

			// SET QUESTION COLOR IF NEEDED
			$messageQuestion = $this->getWrappedQuestionMessage($messageQuestion);
		}

		return $messageQuestion;
	}

	public function getWrappedQuestionMessage($question)
	{
		if ($this->Form)
		{
			return $this->Form->getWrappedQuestionMessage($question);
		}

		return $question;
	}

	public function getWrappedAnswerMessage($answer)
	{
		if ($this->Form && $this->type !== 'wysiwyg')
		{
			return $this->Form->getWrappedAnswerMessage($answer);
		}

		return $answer;
	}

	public function getFormSubmitExtraCost($answer)
	{
		if ($this->type == 'billable_row' && !empty($this->type_data['adds_submission_cost']))
		{
			$cost = $this->type_data['cost'] ?? 1;
			return $cost * intval($answer);
		}

		return 0;
	}

	public function canTriggerConditionals()
	{
		return in_array($this->type, [
			'yes_no',
			'radio',
			'checkboxes',
			'select',
			'multi_select',
			'date_input',
			'datetime_input',
			'time_input',
		]);
	}

	public function getAnswerErrors($answer, $conditionalAnswer = '')
	{
		$errors = [];

		if (in_array($this->type, ['date_input', 'datetime_input', 'time_input']))
		{
			if (is_array($answer))
			{
				$answer = implode(' ', $answer);
			}

			$visitor = \XF::visitor();
			$answerDate = strtotime("$answer $visitor->timezone");
			if ($conditionalAnswer)
			{
				if (is_array($conditionalAnswer))
				{
					$conditionalAnswer = implode(' ', $conditionalAnswer);
				}

				$conditionalAnswerDate = strtotime("$conditionalAnswer $visitor->timezone");

				if ($this->conanswer == 'greater' && $answerDate < $conditionalAnswerDate)
				{
					if (in_array($this->type, ['date_input', 'datetime_input']))
					{
						$error = \XF::phrase('snog_forms_please_enter_a_date_later_than_x', [
							'date' => $this->app()->templater()->func('date_time', [$conditionalAnswerDate])
						]);
					}
					else
					{
						$error = \XF::phrase('snog_forms_please_enter_a_time_later_than_x', [
							'time' => $this->app()->templater()->func('time', [$conditionalAnswerDate])
						]);
					}

					$errors[] = $this->error ?: $error;
				}
				elseif ($this->conanswer == 'lower' && $answerDate > $conditionalAnswerDate)
				{
					if (in_array($this->type, ['date_input', 'datetime_input']))
					{
						$error = \XF::phrase('snog_forms_please_enter_a_date_earlier_than_x', [
							'date' => $this->app()->templater()->func('date_time', [$conditionalAnswerDate])
						]);
					}
					else
					{
						$error = \XF::phrase('snog_forms_please_enter_a_time_earlier_than_x', [
							'time' => $this->app()->templater()->func('time', [$conditionalAnswerDate])
						]);
					}

					$errors[] = $this->error ?: $error;
				}
			}

			if (!empty($this->type_data['date_constraint']))
			{
				if ($this->type_data['date_constraint'] == 'future' && $answerDate < \XF::$time)
				{
					$errors[] = $this->error ?: \XF::phrase('please_enter_a_date_in_the_future');
				}
				elseif ($this->type_data['date_constraint'] == 'past' && $answerDate > \XF::$time)
				{
					$errors[] = $this->error ?: \XF::phrase('please_enter_a_date_in_the_past');
				}
			}
		}
		elseif ($this->type == 'forum_select'
			&& isset($this->type_data['allowed_node_ids'])
			&& !in_array('', $this->type_data['allowed_node_ids'])
			&& !in_array($answer, $this->type_data['allowed_node_ids']))
		{
			$errors[] = $this->error ?: \XF::phrase('snog_forms_this_node_is_not_allowed');
			return $errors;
		}

		// First special case where a default prefix can be set by admin
		// isset required in the event no prefixes can be used
		if (!empty($this->error))
		{
			if ($this->type == 'thread_prefix' && $this->Form && !$this->Form->prefix_ids && !$answer)
			{
				$errors[] = $this->error;
			}
			else
			{
				if ($this->type !== 'thread_prefix')
				{
					$isTextType = in_array($this->type, ['text', 'multiline_text', 'wysiwyg']);

					if (!$isTextType && (!$answer))
					{
						if ($this->type != 'checkbox'
							|| ($this->type == 'checkbox' && $this->expected != ''))
						{
							$errors[] = $this->error;
						}
					}

					if ($isTextType && (!$answer || $answer == ''))
					{
						$errors[] = $this->error;
					}
				}
			}
		}

		// CHECKBOX LIMIT ERRORS
		if ($this->hasCheckboxAnswerError())
		{
			$checkedQuestionCount = is_array($answer) ? count($answer) : 0;
			$checkboxError = $this->getCheckboxAnswerError($answer, $checkedQuestionCount);
			if ($checkboxError)
			{
				$errors[] = $checkboxError;
			}
		}
		elseif ($answer && $this->hasRegexAnswerError())
		{
			$regexError = $this->getRegexAnswerError($answer);
			if ($regexError)
			{
				$errors[] = $regexError;
			}
		}

		$minLength = $this->type_data['minlength'] ?? null;
		$maxLength = $this->type_data['maxlength'] ?? null;

		if ($minLength || $maxLength)
		{
			$answerLength = mb_strlen($answer);

			if ($minLength && $answerLength < $minLength)
			{
				$errors[] = \XF::phrase('please_enter_value_using_x_characters_or_fewer', [
					'count' => $minLength
				]);
			}

			if (isset($maxLength) && $answerLength > $maxLength)
			{
				$errors[] = \XF::phrase('please_enter_value_using_x_characters_or_fewer', [
					'count' => $maxLength
				]);
			}
		}

		if ($this->type == 'billable_row')
		{
			if (!empty($this->type_data['min']) && $answer < $this->type_data['min'])
			{
				$errors[] = \XF::phrase('please_enter_number_that_is_at_least_x', ['min' => $this->type_data['min']]);
			}

			if (!empty($this->type_data['max']) && $answer > $this->type_data['max'])
			{
				$errors[] = \XF::phrase('please_enter_number_that_is_no_more_than_x', ['max' => $this->type_data['max']]);
			}
		}

		return $errors;
	}

	public function hasCheckboxAnswerError()
	{
		return $this->type == 'checkboxes' && ($this->checklimit || $this->checkmin) && !empty($this->checkerror);
	}

	public function getCheckboxAnswerError($answer, $checkedQuestionCount = 0)
	{
		if (($answer > '' && $this->checklimit && $checkedQuestionCount > $this->checklimit)
			|| ($this->checkmin && $checkedQuestionCount < $this->checkmin))
		{
			return $this->checkerror;
		}

		return '';
	}

	public function hasRegexAnswerError()
	{
		return ($this->type == 'text' || $this->type == 'multiline_text' || $this->type == 'spinbox')
			&& !empty($this->regex)
			&& !empty($this->regexerror);
	}

	public function getRegexAnswerError($answer)
	{
		if (is_array($answer))
		{
			foreach ($answer as $answer1)
			{
				if ($this->getRegexAnswerError($answer1))
				{
					return $this->regexerror;
				}
			}
		}

		if ($answer > '' && !preg_match('#' . str_replace('#', '\#', $this->regex) . '#siU', $answer))
		{
			return $this->regexerror;
		}

		return '';
	}

	public function getExportData(): array
	{
		return [
			'questionid' => $this->questionid,
			'posid' => $this->posid,
			'text' => htmlspecialchars($this->text),
			'description' => htmlspecialchars($this->description),
			'type' => $this->type,
			'error' => htmlspecialchars($this->error),
			'expected' => htmlspecialchars($this->expected),
			'display' => $this->display,
			'display_parent' => $this->display_parent,
			'regex' => htmlspecialchars($this->regex),
			'regexerror' => htmlspecialchars($this->regexerror),
			'defanswer' => htmlspecialchars($this->defanswer),
			'readonly' => intval($this->readonly),
			'questionpos' => $this->questionpos,
			'showquestion' => $this->showquestion,
			'showunanswered' => $this->showunanswered,
			'format' => $this->format,
			'hasconditional' => serialize($this->hasconditional),
			'conditional' => $this->conditional,
			'conanswer' => $this->conanswer,
			'placeholder' => htmlspecialchars($this->defanswer),
			'checklimit' => $this->checklimit,
			'checkmin' => $this->checkmin,
			'checkerror' => htmlspecialchars($this->checkerror)
		];
	}

	/************************* LIFE-CYCLE ***************************/

	protected function _preSave()
	{
		if (in_array($this->type, ['radio', 'checkboxes', 'select', 'multi_select']))
		{
			if (!$this->expected)
			{
				$this->error(\XF::phrase('snog_forms_expected_error'));
			}

			if ($this->defanswer)
			{
				$answers = preg_split('/\r?\n/', $this->expected);
				if (!in_array(trim($this->defanswer), $answers))
				{
					$this->error(\XF::phrase('snog_forms_default_error'));
				}
			}
		}

		if ($this->type == 'yes_no' && $this->defanswer)
		{
			$answer1 = \XF::phrase('yes')->render();
			$answer2 = \XF::phrase('no')->render();
			if ($this->defanswer !== $answer1 && $this->defanswer !== $answer2)
			{
				$this->error(\XF::phrase('snog_forms_yesno_error'));
			}
		}

		if ($this->type == 'checkboxes' && $this->checklimit && !$this->checkerror)
		{
			$this->error(\XF::phrase('snog_forms_error_noerror'));
		}

		if ($this->type == 'forum_select' && $this->defanswer)
		{
			$node = $this->finder('XF:Node')->where('title', $this->defanswer)->fetchOne();
			if (!$node)
			{
				$this->error(\XF::phrase('snog_forms_forum_error'));
			}
		}

		/**
		 * if ($this->type !== 'header_phrase' && stristr($this->format, '{question}') === false)
		 * {
		 * $this->error(\XF::phrase('snog_forms_error_question'));
		 * }
		 * if ($this->type !== 'header_phrase' && stristr($this->format, '{answer}') === false)
		 * {
		 * $this->error(\XF::phrase('snog_forms_error_answer'));
		 * }*/

		$conditionalQuestion = $this->ConditionalQuestion;
		if ($conditionalQuestion
			&& in_array($conditionalQuestion->type, ['date_input', 'datetime_input', 'time_input'])
			&& $conditionalQuestion->type != $this->type)
		{
			$this->error(\XF::phrase('snog_forms_date_time_conditionals_only_support_questions_the_same_type'));
		}

		if ($this->type == 'file_upload'
			&& ($this->isInsert() || $this->isChanged('type')))
		{
			$fileQuestions = $this->db()->fetchAll('
				SELECT conditional, conanswer
				FROM xf_snog_forms_questions
				WHERE posid = ? AND type = ?
			', [$this->posid, $this->type]);

			if ($fileQuestions)
			{
				if (!$this->conditional)
				{
					$this->error(\XF::phrase('snog_forms_form_may_contain_only_one_attach_question'));
					return;
				}

				foreach ($fileQuestions as $question)
				{
					if ($this->conditional != $question['conditional'])
					{
						$this->error(\XF::phrase('snog_forms_form_may_contain_only_one_attach_per_conditional_question'));
						return;
					}

					if ($this->conanswer == $question['conanswer'])
					{
						$this->error(\XF::phrase('snog_forms_you_have_already_added_attachment_question_for_this_conditional_answer'));
						return;
					}
				}
			}
		}

		$minLength = $this->type_data['minlength'] ?? null;
		$maxLength = $this->type_data['maxlength'] ?? null;
		if ($minLength && $maxLength && $minLength > $maxLength)
		{
			$this->error(\XF::phrase('snog_forms_min_length_must_be_lower_than_max_length'));
		}
	}

	protected function _postSave()
	{
		// QUESTION CHANGED AND CAN NO LONGER BE USED AS A TRIGGER FOR CONDITIONAL QUESTION
		if (!empty($this->hasconditional) && !$this->canTriggerConditionals())
		{
			// DELETE CONDITIONAL QUESTIONS
			foreach ($this->hasconditional as $conditional)
			{
				/** @var Question $oldConditional */
				$oldConditional = $this->em()->findOne('Snog\Forms:Question', ['questionid', '=', $conditional]);
				if ($oldConditional)
				{
					$oldConditional->delete();
				}
			}

			$this->fastUpdate('hasconditional', []);
		}
	}

	protected function _postDelete()
	{
		$db = $this->db();
		$db->delete("xf_snog_forms_answers", 'questionid = ?', $this->questionid);

		// Remove question from a conditional list in master question
		if ($this->conditional)
		{
			/** @var Question $masterQuestion */
			$masterQuestion = $this->em()->find('Snog\Forms:Question', $this->conditional);
			if ($masterQuestion->exists())
			{
				$existingConditionals = $masterQuestion->hasconditional;
				if (($key = array_search($this->questionid, $existingConditionals)) !== false)
				{
					unset($existingConditionals[$key]);
				}
				$masterQuestion->set('hasconditional', $existingConditionals, ['forceSet' => true]);
				$masterQuestion->save();
			}
		}

		// Remove conditional questions when master question is deleted
		if ($this->hasconditional)
		{
			foreach ($this->hasconditional as $conditional)
			{
				/** @var Question $conditionalQuestion */
				$conditionalQuestion = $this->em()->find('Snog\Forms:Question', $conditional);
				if ($conditionalQuestion)
				{
					$conditionalQuestion->delete();
				}
			}
		}

		// Renumbers question display order after delete
		if ($this->posid)
		{
			$posid = $this->posid;

			$questionRepo = $this->getQuestionRepo();
			$questionList = $questionRepo->getQuestionList($posid);
			$position = 1;

			foreach ($questionList as $question)
			{
				$db->update('xf_snog_forms_questions', ['display' => $position], 'questionid = ?', $question['questionid']);
				$position++;
			}
		}
	}

	public static function getStructure(Structure $structure)
	{
		$structure->table = 'xf_snog_forms_questions';
		$structure->shortName = 'Snog\Forms:Question';
		$structure->contentType = 'snog_forms_question';
		$structure->primaryKey = 'questionid';
		$structure->columns = [
			'questionid' => ['type' => self::UINT, 'autoIncrement' => true, 'nullable' => true],
			'posid' => ['type' => self::UINT, 'default' => 0],
			'text' => ['type' => self::STR, 'required' => 'snog_forms_question_text_error'],
			'description' => ['type' => self::STR, 'maxLength' => 250, 'default' => ''],
			'type' => ['type' => self::STR, 'maxLength' => 50, 'default' => 'text'],
			'error' => ['type' => self::STR, 'maxLength' => 200, 'default' => ''],
			'expected' => ['type' => self::STR],
			'display' => ['type' => self::UINT, 'default' => 0],
			'display_parent' => ['type' => self::UINT, 'default' => 0],
			'regex' => ['type' => self::STR, 'default' => ''],
			'regexerror' => ['type' => self::STR, 'maxLength' => 200, 'default' => ''],
			'defanswer' => ['type' => self::STR, 'maxLength' => 200, 'default' => ''],
			'readonly' => ['type' => self::BOOL, 'default' => 0],
			'questionpos' => ['type' => self::UINT],
			'showquestion' => ['type' => self::UINT, 'default' => 1],
			'showunanswered' => ['type' => self::UINT, 'default' => 1],
			'inline' => ['type' => self::UINT, 'default' => 1],
			'format' => ['type' => self::STR, 'default' => ''],
			'hasconditional' => ['type' => self::JSON_ARRAY, 'default' => []],
			'conditional' => ['type' => self::UINT, 'default' => 0],
			'conanswer' => ['type' => self::STR, 'default' => ''],
			'placeholder' => ['type' => self::STR, 'maxLength' => 200, 'default' => ''],
			'checklimit' => ['type' => self::UINT, 'default' => 0],
			'checkmin' => ['type' => self::UINT, 'default' => 0],
			'checkerror' => ['type' => self::STR, 'maxLength' => 200, 'default' => ''],
			'type_data' => ['type' => self::JSON_ARRAY, 'default' => []]
		];

		$structure->getters = [];
		$structure->relations = [
			'Form' => [
				'entity' => 'Snog\Forms:Form',
				'type' => self::TO_ONE,
				'conditions' => 'posid',
				'primary' => true
			],
			'ConditionalQuestion' => [
				'entity' => 'Snog\Forms:Question',
				'type' => self::TO_ONE,
				'conditions' => [['questionid', '=', '$conditional']],
				'primary' => true
			],
		];

		$structure->options['is_first'] = false;

		return $structure;
	}

	/**
	 * @return \Snog\Forms\Repository\Question|\XF\Mvc\Entity\Repository
	 */
	protected function getQuestionRepo()
	{
		return $this->repository('Snog\Forms:Question');
	}
}
