<?php

namespace Snog\Forms\Entity;

use XF\Mvc\Entity\Entity;
use XF\Mvc\Entity\Structure;

/**
 * COLUMNS
 * @property int $purchase_id
 * @property int $form_id
 * @property int $user_id
 * @property string $username
 * @property string $ip_address
 * @property string $purchase_state
 * @property int $purchase_date
 * @property string $purchase_request_key
 * @property float $cost_amount
 * @property string $cost_currency
 * @property array|null $extra_data
 *
 * GETTERS
 * @property string $purchasable_type_id
 * @property string $purchase_state_phrase
 *
 * RELATIONS
 * @property Form $Form
 * @property \XF\Entity\User $User
 * @property \XF\Entity\PurchaseRequest $PurchaseRequest
 * @property \XF\Entity\PaymentProviderLog $PaymentProviderLog
 */
class SubmitPurchase extends Entity
{
	public function getPurchaseStatePhrase()
	{
		return \XF::phrase('snog_forms_purchase_state.' . $this->purchase_state);
	}

	/**
	 * @return string
	 */
	public function getPurchasableTypeId()
	{
		return 'snog_forms_form_submit';
	}

	/**
     * @param Structure $structure
     * @return Structure
     */
    public static function getStructure(Structure $structure)
    {
        $structure->table = 'xf_snog_forms_submit_purchase';
        $structure->shortName = 'Snog\Forms:SubmitPurchase';
        $structure->primaryKey = 'purchase_id';
        $structure->columns = [
            'purchase_id' => ['type' => static::UINT, 'autoIncrement' => true],
            'form_id' => ['type' => static::UINT, 'required' => true],
            'user_id' => ['type' => static::UINT, 'required' => true],
            'username' => ['type' => static::STR, 'maxLength' => 50, 'default' => ''],
			'ip_address' => ['type' => static::BINARY, 'maxLength' => 16, 'required' => true],
            'purchase_state' => ['type' => static::STR, 'allowedValues' => ['awaiting_payment', 'valid', 'refunded'], 'default' => 'awaiting_payment'],
            'purchase_date' => ['type' => static::UINT, 'default' => '0'],
            'purchase_request_key' => ['type' => static::STR, 'maxLength' => 32, 'nullable' => true, 'default' => null],
            'cost_amount' => ['type' => static::FLOAT, 'default' => '0.00', 'min' => 0.00],
            'cost_currency' => ['type' => static::STR, 'maxLength' => 3, 'default' => ''],
            'extra_data' => ['type' => static::JSON_ARRAY, 'default' => []],
        ];

		$structure->getters = [
			'purchase_state_phrase' => true
		];

		$structure->relations = [
			'Form' => [
				'entity' => 'Snog\Forms:Form',
				'type' => self::TO_ONE,
				'conditions' => [['posid', '=', '$form_id']],
				'primary' => true
			],
			'User' => [
				'entity' => 'XF:User',
				'type' => self::TO_ONE,
				'conditions' => 'user_id',
				'primary' => true,
			],
			'PurchaseRequest' => [
				'entity' => 'XF:PurchaseRequest',
				'type' => self::TO_ONE,
				'conditions' => [
					['request_key', '=', '$purchase_request_key']
				],
				'primary' => true,
			],
			'PaymentProviderLog' => [
				'entity' => 'XF:PaymentProviderLog',
				'type' => self::TO_ONE,
				'conditions' => 'purchase_request_key',
				'primary' => true,
			]
		];

		$structure->getters['purchasable_type_id'] = true;

        return $structure;
    }
}