<?php

namespace Snog\Forms\Mvc\Controller;

use Snog\Forms\Entity\Form as FormEntity;
use XF\Mvc\ParameterBag;

trait LogListTrait
{
	abstract protected function getFormLogClassIdentifier();

	protected function getFormLogLinkType()
	{
		return 'public';
	}

	abstract protected function getFormLogLinkPrefix();

	abstract protected function getFormLogTemplatePrefix();

	public function actionIndex()
	{
		$logRepo = $this->getFormsLogRepo();
		$logFinder = $logRepo->findLogs()
			->with(['User', 'Form'])
			->orderByDate('DESC');

		$this->adjustLogFinder($logFinder);

		$filters = $this->getLogFilterInput();
		$this->applyLogFilters($logFinder, $filters);

		$page = $this->filterPage();
		$logFinder->limitByPage($page, 20);
		$total = $logFinder->total();

		$viewParams = [
			'logs' => $logFinder->fetch(),
			'link' => $this->getFormLogLinkPrefix(),
			'linkType' => $this->getFormLogLinkType(),
			'total' => $total,
			'page' => $page,
			'perPage' => 20,
			'filters' => $filters
		];

		return $this->view(
			$this->getFormLogClassIdentifier() . '\Listing',
			$this->getFormLogTemplatePrefix() . '_list',
			$viewParams
		);
	}

	public function actionView(ParameterBag $params)
	{
		$entry = $this->assertFormsLogExists($params->log_id);

		/** @var \Snog\Forms\Repository\Answer $answerRepo */
		$answerRepo = $this->repository('Snog\Forms:Answer');
		$answers = $answerRepo->findAnswers()->with('Question')->byLog($entry->log_id)->fetch();

		$viewParams = [
			'entry' => $entry,
			'answers' => $answers,
			'link' => $this->getFormLogLinkPrefix(),
			'linkType' => $this->getFormLogLinkType(),
		];

		return $this->view(
			$this->getFormLogClassIdentifier() . '\View',
			$this->getFormLogTemplatePrefix() . '_view',
			$viewParams
		);
	}

	public function adjustLogFinder(\Snog\Forms\Finder\Log $logFinder)
	{
	}

	public function actionRefineSearch()
	{
		$filters = $this->getLogFilterInput();

		if ($this->filter('apply', 'bool'))
		{
			return $this->redirect($this->buildLink($this->getFormLogLinkPrefix(), null, $filters));
		}

		/** @var \Snog\Forms\Repository\Form $formRepo */
		$formRepo = $this->repository('Snog\Forms:Form');

		/** @var \XF\Mvc\Entity\ArrayCollection|FormEntity[] $formValues */
		$forms = $formRepo->findActiveFormsForList()->fetch()->filterViewable();

		$viewParams = [
			'filters' => $filters,
			'forms' => $forms,
			'link' => $this->getFormLogLinkPrefix(),
			'linkType' => $this->getFormLogLinkType(),
			'datePresets' => $this->app->language()->getDatePresets(),
			'sortOptions' => $this->getAvailableLogSorts()
		];

		return $this->view(
			$this->getFormLogClassIdentifier() . '\RefineSearch',
			$this->getFormLogTemplatePrefix() . '_refine_search',
			$viewParams
		);
	}

	protected function getLogFilterInput()
	{
		$filters = [];

		$input = $this->filter([
			'username' => 'str',
			'form_id' => 'uint',
			'order' => 'str',
			'direction' => 'str',
			'start' => 'datetime',
			'end' => 'datetime',
		]);

		if ($input['username'])
		{
			$filters['username'] = $input['username'];
		}

		if ($input['form_id'])
		{
			$filters['form_id'] = $input['form_id'];
		}

		if ($input['start'])
		{
			$filters['start'] = $input['start'];
		}

		if ($input['end'])
		{
			$filters['end'] = $input['end'];
		}

		$sorts = $this->getAvailableLogSorts();

		if ($input['order'] && isset($sorts[$input['order']]))
		{
			if (!in_array($input['direction'], ['asc', 'desc']))
			{
				$input['direction'] = 'asc';
			}

			if ($input['order'] != 'log_date' || $input['direction'] != 'asc')
			{
				$filters['order'] = $input['order'];
				$filters['direction'] = $input['direction'];
			}
		}

		return $filters;
	}

	/**
	 * @return string[]
	 */
	protected function getAvailableLogSorts()
	{
		return [
			'log_date' => 'log_date'
		];
	}

	protected function applyLogFilters(\XF\Mvc\Entity\Finder $finder, array $filters)
	{
		if (!empty($filters['username']))
		{
			$finder->with('User');

			$expression = $finder->escapeLike($filters['username'], '%?%');
			$finder->whereOr(['User.username', 'LIKE', $expression], ['username', 'LIKE', $expression]);
		}

		if (!empty($filters['form_id']))
		{
			$finder->where('form_id', '=', $filters['form_id']);
		}

		if (!empty($filters['start']))
		{
			$finder->where('log_date', '>', $filters['start']);
		}

		if (!empty($filters['end']))
		{
			$finder->where('log_date', '<', $filters['end']);
		}

		$sorts = $this->getAvailableLogSorts();
		if (!empty($filters['order']) && isset($sorts[$filters['order']]))
		{
			$finder->order($sorts[$filters['order']], $filters['direction']);
		}
	}

	/**
	 * @param string $id
	 * @param array|string|null $with
	 * @param null|string $phraseKey
	 *
	 * @return \Snog\Forms\Entity\Log|\XF\Mvc\Entity\Entity
	 * @throws \Exception
	 */
	protected function assertFormsLogExists($id, $with = null, $phraseKey = null)
	{
		return $this->assertRecordExists('Snog\Forms:Log', $id, $with, $phraseKey);
	}

	/**
	 * @return \Snog\Forms\Repository\Log|\XF\Mvc\Entity\Repository
	 */
	protected function getFormsLogRepo()
	{
		return $this->repository('Snog\Forms:Log');
	}
}