<?php

namespace Snog\Forms\Pub\Controller;

use Snog\Forms\Entity\Form as FormEntity;
use Snog\Forms\Service\Form\Submit;
use XF;
use XF\Entity\Thread;
use XF\Mvc\ParameterBag;
use XF\Pub\Controller\AbstractController;

class Form extends AbstractController
{
	public function actionIndex()
	{
		$formRepo = $this->getFormRepo();

		/** @var \XF\Mvc\Entity\ArrayCollection|FormEntity[] $formValues */
		$formValues = $formRepo->findActiveFormsForList()->fetch()->filterViewable();

		$headerValues = [];

		// EXTRACT FORM TYPES FOR DISPLAY

		/** @var FormEntity $formHeader */
		foreach ($formValues as $formHeader)
		{
			if (isset($formHeader->Type->type) && !in_array($formHeader->Type->type, $headerValues))
			{
				$headerValues[] = $formHeader->Type->type;
			}
			if (empty($formHeader->Type))
			{
				$headerValues[] = null;
			}
		}

		$viewParams = [
			'forms' => $formValues,
			'headervalues' => $headerValues,
			'purchaseSuccess' => $this->filter('purchase_success', 'bool')
		];

		return $this->view('Snog:Forms\Form', 'snog_forms_list', $viewParams);
	}

	/**
	 * @param ParameterBag $params
	 * @return XF\Mvc\Reply\View
	 * @throws XF\Mvc\Reply\Exception
	 */
	public function actionSelect(ParameterBag $params)
	{
		$this->app->response()->header('Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0');

		$form = $this->assertViewableForm($params['posid'], $this->getFormViewExtraWith());

		/** @var \Snog\Forms\XF\Entity\User $visitor */
		$visitor = XF::visitor();
		$replyThread = null;

		// GET THREAD ID FROM REPLY BUTTON IF NEEDED
		if ($form->qroption)
		{
			$replyThread = $this->filter('thread', 'uint');
		}

		/** @var \Snog\Forms\ControllerPlugin\Form $formPlugin */
		$formPlugin = $this->plugin('Snog\Forms:Form');
		$formPlugin->assertUsableFormFromContext($form, $visitor, $replyThread);

		// GET QUESTIONS
		$questionRepo = $this->getQuestionRepo();
		$questions = $questionRepo->getQuestionList($form->posid);
		$uploadNeeded = false;
		$forum = '';
		$prefixes = [];
		$conditionQuestions = [];

		$nodeId = $this->filter('node_id', 'uint');
		$hasForumSelectQuestion = false;

		// SETUP CONDITIONALS AND ANSWERS FOR CHOICE QUESTIONS
		foreach ($questions as $question)
		{
			// BUILD AND SORT CONDITIONAL QUESTIONS
			if ($question->hasconditional)
			{
				$conditionals = $questionRepo->getQuestionConditionals($question, $questions, $uploadNeeded);
				$conditionQuestions[$question['questionid']] = $conditionals;
			}

			// DETECT FILE UPLOAD QUESTION TYPE
			if ($question->isUploadNeeded())
			{
				$uploadNeeded = true;
			}

			// Select referring forum
			if ($question->type == 'forum_select' && empty($question->defanswer) && $nodeId)
			{
				$question->defanswer = $nodeId;
				$hasForumSelectQuestion = true;
			}
		}

		$forceNodeId = $this->filter('node_id', 'uint');
		if ($forceNodeId && !$hasForumSelectQuestion)
		{
			/** @var \XF\Entity\Forum $forum */
			$forum = $this->em()->find('XF:Forum', $forceNodeId);
			if ($forum->canView($error) && in_array($forceNodeId, $form->post_thread_button_node_ids))
			{
				$form->setOption('report_node_id', $forceNodeId);
			}
			else
			{
				$forum = null;
			}
		}

		$warnings = [];
		$canSubmit = $form->canSubmit();
		if (!$canSubmit)
		{
			$warnings[] = XF::phrase('snog_forms_not_allowed_to_submit');
		}

		if ($form->cooldown !== 0)
		{
			$cooldownError = $formPlugin->assertFormCooldown($form, $visitor, $this->request->getIp(), false);
			if (!empty($cooldownError))
			{
				$canSubmit = false;
				$warnings[] = $cooldownError;
			}
		}

		// SETUP FOR FILE UPLOAD
		$noUpload = false;
		$attachmentData = null;

		if ($uploadNeeded && $canSubmit && (($form->inthread && ($form->node_id || $forceNodeId)) || $form->bypm || $form->oldthread || $replyThread))
		{
			if ($form->inthread && ($form->node_id || $forceNodeId))
			{
				$poster = $form->getPoster($visitor);
				$forum = $forum ?: $form->Forum;
				if ($forum)
				{
					$prefixes = $forum->getUsablePrefixes();

					if (!$forum->canCreateThreadFromForm($poster) || !$forum->canUploadAndManageAttachmentsFromForm($poster))
					{
						$noUpload = true;
					}

					$attachmentRepo = $this->getAttachmentRepo();
					$attachmentData = $attachmentRepo->getEditorData('post', $forum, $forum->draft_thread['attachment_hash']);
				}
			}
			else
			{
				if ($form->oldthread || $replyThread)
				{
					$thread = $this->assertExistingThread($replyThread, $form);

					$attachmentHash = $thread->draft_reply->attachment_hash;
					$attachmentRepo = $this->getAttachmentRepo();
					$attachmentData = $attachmentRepo->getEditorData('post', $thread, $attachmentHash);
				}
				elseif ($form->bypm)
				{
					$draft = \XF\Draft::createFromKey('conversation');
					$attachmentRepo = $this->getAttachmentRepo();
					$attachmentData = $attachmentRepo->getEditorData('conversation_message', null, $draft->attachment_hash);
				}
			}
		}

		// GET PREFIXES IF FORUM NOT ALREADY DEFINED
		if (!$forum && $form->node_id)
		{
			$forum = $form->Forum;
			if ($forum)
			{
				$prefixes = $forum->getUsablePrefixes();
			}
		}

		// CHANGE THE STYLE FOR THIS FORM IF NEEDED
		if ($form->app_style)
		{
			$this->setViewOption('style_id', $form->app_style);
		}

		/** @var \XF\Repository\Node $nodeRepo */
		$nodeRepo = $this->repository('XF:Node');
		$nodeTree = $nodeRepo->createNodeTree($nodeRepo->getFullNodeList());

		$viewParams = [
			'form' => $form,
			'forum' => $forum,
			'replythread' => $replyThread,
			'prefixes' => $prefixes,
			'noupload' => $noUpload,
			'questions' => $questions,
			'conditionQuestions' => $conditionQuestions,
			'nodeTree' => $nodeTree,
			'attachmentData' => $attachmentData,
			'canSubmit' => $canSubmit,
			'warnings' => $warnings,
			'purchaseSuccess' => $this->filter('purchase_success', 'bool')
		];

		return $this->view('Snog:Forms\Form', 'snog_forms_form', $viewParams);
	}

	protected function getFormViewExtraWith()
	{
		return ['Type', 'ConversationUser', 'Forum', 'SecondForum'];
	}

	/**
	 * @param ParameterBag $params
	 * @return XF\Mvc\Reply\AbstractReply|XF\Mvc\Reply\Error|XF\Mvc\Reply\Redirect
	 * @throws XF\Mvc\Reply\Exception
	 * @throws XF\PrintableException
	 */
	public function actionSubmit(ParameterBag $params)
	{
		$this->assertPostOnly();

		$form = $this->assertFormExists($params['posid'], [
			'Type',
			'PosterUser',
			'SecondaryPosterUser',
			'ConversationUser',
			'Forum',
			'SecondForum'
		]);

		$replyThread = null;
		$replyThreadId = $this->filter('replythread', 'uint');

		// GET THREAD ID FROM COMPLETED FORM IF NEEDED
		if ($replyThreadId && $form->qroption)
		{
			/** @var Thread $replyThread */
			$replyThread = $this->em()->findOne('XF:Thread', ['thread_id', '=', $replyThreadId]);
			if (!$replyThread)
			{
				return $this->noPermission();
			}
		}

		/** @var \Snog\Forms\ControllerPlugin\Form $formPlugin */
		$formPlugin = $this->plugin('Snog\Forms:Form');
		$formPlugin->assertUsableFormFromContext($form, $replyThread);

		if (!$form->canSubmit($error))
		{
			return $this->noPermission($error);
		}

		$this->assertNotFlooding('form', $this->options()->ozzmodz_forms_floodCheck);

		/** @var \Snog\Forms\XF\Entity\User $user */
		$user = XF::visitor();

		if (!$this->captchaIsValid())
		{
			return $this->error(XF::phrase('did_not_complete_the_captcha_verification_properly'));
		}

		$request = $this->app()->request();
		$ip = $request->getIp();

		if ($form->cooldown !== 0)
		{
			$formPlugin->assertFormCooldown($form, $user, $ip);
		}

		$submitter = $this->setupFormSubmitter($form, $replyThread);
		if (!$submitter->validate($errors))
		{
			return $this->error($errors);
		}

		$extraCost = $submitter->getAnswersExtraCost();
		$submitCost = $form->cost_amount + $extraCost;

		if ($form->isPurchasable() && $submitCost > 0)
		{
			if (!$form->canPurchase($error))
			{
				return $this->noPermission($error);
			}

			$formData = $this->filter([
				'question' => 'array',
				'attachment_hash' => 'str'
			]);

			$formData['extra_cost'] = $extraCost;

			$submitPurchase = $form->getNewPurchase($user, $submitCost, $formData);
			if ($submitPurchase)
			{
				$submitPurchase->save();
			}

			/** @var \XF\Repository\Payment $paymentRepo */
			$paymentRepo = $this->repository('XF:Payment');
			$profiles = $paymentRepo->getPaymentProfileOptionsData();

			$allowedProfileIds = $form->getAllowedPurchasePaymentProfileIds();
			if (empty($allowedProfileIds))
			{
				return $this->error(\XF::phrase('this_item_cannot_be_purchased_at_moment'));
			}

			$viewParams = [
				'purchase' => $submitPurchase,
				'form' => $form,
				'profiles' => $profiles,
				'enabledProfileIds' => $allowedProfileIds
			];

			return $this->view('Snog:Forms\Form\Purchase', 'snog_forms_purchase_form', $viewParams);
		}
		else
		{
			$submitter->save();

			$thread = null;
			$returnPost = null;
			$threadCreator = $submitter->getThreadCreator();
			if ($threadCreator)
			{
				$thread = $threadCreator->getThread();
			}
			else
			{
				$secondThreadCreator = $submitter->getSecondThreadCreator();
				if ($secondThreadCreator)
				{
					$thread = $secondThreadCreator->getThread();
				}
			}
			$replier = $submitter->getReplier();
			if ($replier)
			{
				$returnPost = $replier->getPost();
			}

			$redirectContext = [
				'thread' => $thread,
				'post' => $returnPost,
			];

			// Safety jic something goes wrong with returnto
			return $this->redirect($form->getRedirectUrl($redirectContext), $form->thanks);
		}
	}

	/**
	 * @param $form
	 * @param Thread|null $replyThread
	 * @return Submit
	 */
	protected function setupFormSubmitter(FormEntity $form, ?Thread $replyThread = null)
	{
		// GET QUESTIONS
		$questionRepo = $this->getQuestionRepo();
		$questions = $questionRepo->getQuestionList($form->posid);

		/** @var Submit $submitter */
		$submitter = $this->service('Snog\Forms:Form\Submit', $form, $questions);
		$submitter->setAttachmentHash($this->filter('attachment_hash', 'str'));
		$submitter->setAnswers($this->filter('question', 'array'));

		if ($replyThread)
		{
			$submitter->setReplyThread($replyThread);
		}

		return $submitter;
	}

	/**
	 * @param ParameterBag $params
	 * @return XF\Mvc\Reply\AbstractReply|XF\Mvc\Reply\Redirect|XF\Mvc\Reply\View
	 * @throws XF\Mvc\Reply\Exception
	 */
	public function actionApprove(ParameterBag $params)
	{
		/** @var \Snog\Forms\XF\Entity\User $visitor */
		$visitor = XF::visitor();
		if (!$visitor->canApproveAdvancedForms($error))
		{
			return $this->noPermission($error);
		}

		$postID = $params['posid'];
		$promotion = $this->assertPromotionExists($postID, ['Form', 'Form.ConversationUser', 'Thread', 'User']);

		if ($this->isPost())
		{
			/** @var \Snog\Forms\Service\Promotion\Approve $approver */
			$approver = $this->service('Snog\Forms:Promotion\Approve', $promotion);
			if (!$approver->validate($errors))
			{
				return $this->error($errors);
			}
			$approver->approve();

			return $this->redirect($this->buildLink('threads', $promotion->Thread));
		}

		/** @var Thread $thread */
		$thread = $this->em()->findOne('XF:Thread', ['first_post_id', '=', $postID]);
		$viewParams = ['approve' => $promotion, 'title' => $thread->title];
		return $this->view('Snog:Forms\Form', 'snog_forms_confirm', $viewParams);
	}

	/**
	 * @param ParameterBag $params
	 * @return XF\Mvc\Reply\AbstractReply|XF\Mvc\Reply\Redirect|XF\Mvc\Reply\View
	 * @throws XF\Mvc\Reply\Exception
	 */
	public function actionDeny(ParameterBag $params)
	{
		/** @var \Snog\Forms\XF\Entity\User $visitor */
		$visitor = XF::visitor();
		if (!$visitor->canApproveAdvancedForms($error))
		{
			return $this->noPermission($error);
		}

		$postID = $params['posid'];
		$promotion = $this->assertPromotionExists($postID, ['Form', 'Form.ConversationUser', 'Thread']);

		if ($this->isPost())
		{
			/** @var \Snog\Forms\Service\Promotion\Deny $denier */
			$denier = $this->service('Snog\Forms:Promotion\Deny', $promotion);
			if (!$denier->validate($errors))
			{
				return $this->error($errors);
			}
			$denier->deny();

			return $this->redirect($this->buildLink('threads', $promotion->Thread));
		}

		/** @var Thread $thread */
		$thread = $this->em()->findOne('XF:Thread', ['first_post_id', '=', $postID]);
		$viewParams = ['deny' => $promotion, 'title' => $thread->title];
		return $this->view('Snog:Forms\Form', 'snog_forms_confirm', $viewParams);
	}

	/**
	 * @param ParameterBag $params
	 * @return XF\Mvc\Reply\AbstractReply|XF\Mvc\Reply\Redirect
	 * @throws XF\Mvc\Reply\Exception
	 * @throws XF\PrintableException
	 */
	public function actionExtend(ParameterBag $params)
	{
		/** @var \Snog\Forms\XF\Entity\User $visitor */
		$visitor = XF::visitor();
		if (!$visitor->canExtendAdvancedFormsPolls($error))
		{
			return $this->noPermission($error);
		}

		$postID = $params['posid'];
		$extend = $this->assertPromotionExists($postID, ['Form', 'Thread']);

		/** @var \XF\Entity\Poll $poll */
		$poll = $this->em()->find('XF:Poll', $extend->poll_id);
		$close_date = $poll->close_date;
		$poll->close_date = $close_date + 86400;
		if ($poll->isChanged('close_date'))
		{
			$poll->save(false, false);
		}

		$extend->close_date = $close_date + 86400;
		if ($extend->isChanged('close_date'))
		{
			$extend->save(false, false);
		}

		return $this->redirect($this->buildLink('threads', $extend->Thread));
	}

	/**
	 * @param $id
	 * @param $with
	 * @return FormEntity|XF\Mvc\Entity\Entity
	 * @throws XF\Mvc\Reply\Exception
	 */
	protected function assertViewableForm($id, $with = [])
	{
		$form = $this->assertFormExists($id, $with);
		if (!$form)
		{
			throw $this->exception($this->notFound());
		}
		if (!$form->canView($error))
		{
			throw $this->exception($this->noPermission($error));
		}

		return $form;
	}

	/**
	 * @param $id
	 * @param null $with
	 * @return FormEntity|\XF\Mvc\Entity\Entity
	 * @throws \XF\Mvc\Reply\Exception
	 */
	protected function assertFormExists($id, $with = null)
	{
		return $this->assertRecordExists('Snog\Forms:Form', $id, $with, 'snog_forms_form_not_found');
	}

	/**
	 * @param $id
	 * @param null $with
	 * @return \XF\Mvc\Entity\Entity|\Snog\Forms\Entity\Promotion
	 * @throws \XF\Mvc\Reply\Exception
	 */
	protected function assertPromotionExists($id, $with = null)
	{
		return $this->assertRecordExists('Snog\Forms:Promotion', $id, $with, 'snog_forms_promotion_not_found');
	}

	/**
	 * @param $conversationId
	 * @param $sender
	 * @param array $extraWith
	 * @return \XF\Entity\ConversationUser|\XF\Mvc\Entity\Entity
	 * @noinspection PhpReturnDocTypeMismatchInspection
	 * @throws \XF\Mvc\Reply\Exception
	 */
	protected function assertViewableUserConversation($conversationId, $sender, array $extraWith = [])
	{
		/** @var \XF\Finder\ConversationUser $finder */
		$finder = $this->finder('XF:ConversationUser');

		$conversation = $finder->forUser($sender, false)
			->where('conversation_id', $conversationId)
			->with($extraWith)
			->fetchOne();

		if (!$conversation)
		{
			throw $this->exception($this->notFound(XF::phrase('requested_conversation_not_found')));
		}

		return $conversation;
	}

	public static function getActivityDetails(array $activities)
	{
		$formData = [];
		$posIds = [];
		$router = XF::app()->router('public');

		foreach ($activities as $activity)
		{
			$posId = $activity->pluckParam('posid');
			if ($posId) $posIds[$posId] = $posId;
		}

		if ($posIds)
		{
			/** @var FormEntity[] $forms */
			$forms = XF::em()->findByIds('Snog\Forms:Form', $posIds);
			foreach ($forms as $id => $form)
			{
				$formData[$id] = ['position' => $form->position, 'url' => $router->buildLink('form')];
			}
		}

		$output = [];
		foreach ($activities as $key => $activity)
		{
			$posId = $activity->pluckParam('posid');
			$form = $posId && isset($formData[$posId]) ? $formData[$posId] : null;
			if ($form)
			{
				$output[$key] = ['description' => XF::phrase('snog_forms_filling_out'), 'title' => $form['position'], 'url' => $form['url'],];
			}
			else
			{
				$output[$key] = XF::phrase('snog_forms_viewing_list');
			}
		}

		return $output;
	}

	/**
	 * @param ParameterBag $params
	 * @return false|XF\Mvc\Reply\AbstractReply|XF\Mvc\Reply\View
	 * @throws XF\Mvc\Reply\Exception
	 */
	public function actionAnswerPreview(ParameterBag $params)
	{
		if (!$this->options()->ozzmodz_forms_bbCodePreview)
		{
			return false;
		}

		$this->assertPostOnly();

		$form = $this->assertViewableForm($params->posid);
		if (!$form->canSubmit($error))
		{
			return $this->noPermission($error);
		}

		$questionId = $this->filter('question_id', 'uint');

		/** @var \XF\ControllerPlugin\Editor $editorPlugin */
		$editorPlugin = $this->plugin('XF:Editor');
		$message = $editorPlugin->fromInput("question.$questionId");

		/** @var \XF\ControllerPlugin\BbCodePreview $previewPlugin */
		$previewPlugin = $this->plugin('XF:BbCodePreview');
		return $previewPlugin->actionPreview(
			$message, 'snog_form'
		);
	}

	/**
	 * @param ParameterBag $params
	 * @return XF\Mvc\Reply\AbstractReply
	 * @throws XF\Mvc\Reply\Exception
	 */
	public function actionBookmark(ParameterBag $params)
	{
		$form = $this->assertViewableForm($params->posid);

		/** @var \XF\ControllerPlugin\Bookmark $bookmarkPlugin */
		$bookmarkPlugin = $this->plugin('XF:Bookmark');

		return $bookmarkPlugin->actionBookmark(
			$form, $this->buildLink('form/bookmark', $form)
		);
	}

	public function assertNotBanned()
	{
		// TODO
		parent::assertNotBanned();
	}

	/**
	 * @param $replyThread
	 * @param FormEntity $form
	 * @return Thread
	 * @throws XF\Mvc\Reply\Exception
	 */
	protected function assertExistingThread($replyThread, FormEntity $form)
	{
		if ($replyThread)
		{
			/** @var Thread $thread */
			$thread = $this->em()->findOne('XF:Thread', ['thread_id', '=', $replyThread], ['Forum', 'Forum.Node']);
		}
		else
		{
			/** @var Thread $thread */
			$thread = $this->em()->findOne('XF:Thread', ['thread_id', '=', $form->oldthread], ['Forum', 'Forum.Node']);
		}

		if (!$thread)
		{
			throw $this->exception($this->notFound(XF::phrase('requested_thread_not_found')));
		}

		return $thread;
	}

	/**
	 * @return \XF\Repository\Attachment|\XF\Mvc\Entity\Repository
	 */
	protected function getAttachmentRepo()
	{
		return $this->repository('XF:Attachment');
	}

	/**
	 * @return \Snog\Forms\Repository\Question|\XF\Mvc\Entity\Repository
	 */
	protected function getQuestionRepo()
	{
		return $this->repository('Snog\Forms:Question');
	}

	/**
	 * @return \Snog\Forms\Repository\Form|\XF\Mvc\Entity\Repository
	 */
	protected function getFormRepo()
	{
		return $this->repository('Snog\Forms:Form');
	}


}