<?php

namespace Snog\Forms\Purchasable;

use XF\Entity\PaymentProfile;
use XF\Entity\Thread;
use XF\Entity\User;
use XF\Http\Request;
use XF\Payment\CallbackState;
use XF\Purchasable\AbstractPurchasable;
use XF\Purchasable\Purchase;

class FormSubmit extends AbstractPurchasable
{
	public function getTitle()
	{
		return \XF::phrase('snog_forms_form_submit');
	}

	public function getPurchaseFromRequest(Request $request, User $purchaser, &$error = null)
	{
		$profileId = $request->filter('payment_profile_id', 'uint');

		/** @var PaymentProfile $paymentProfile */
		$paymentProfile = \XF::em()->find('XF:PaymentProfile', $profileId);
		if (!$paymentProfile || !$paymentProfile->active)
		{
			$error = \XF::phrase('please_choose_valid_payment_profile_to_continue_with_your_purchase');
			return false;
		}

		$purchaseId = $request->filter('purchase_id', 'uint');

		/** @var \Snog\Forms\Entity\SubmitPurchase $submitPurchase */
		$submitPurchase = \XF::em()->find('Snog\Forms:SubmitPurchase', $purchaseId);
		if (!$submitPurchase)
		{
			$error = \XF::phrase('this_item_cannot_be_purchased_at_moment');
			return false;
		}

		$form = $submitPurchase->Form;
		if (!$form || !$form->canPurchase($error))
		{
			if (!$error)
			{
				$error = \XF::phrase('this_item_cannot_be_purchased_at_moment');
			}
			return false;
		}

		if (!$form->isAllowedPurchasePaymentProfile($profileId))
		{
			$error = \XF::phrase('selected_payment_profile_is_not_valid_for_this_purchase');
			return false;
		}

		return $this->getPurchaseObject($paymentProfile, $submitPurchase, $purchaser);
	}

	public function validatePurchasable(CallbackState $state, &$error = null): bool
	{
		$purchaseRequest = $state->getPurchaseRequest();
		$extraData = $purchaseRequest->extra_data;

		/** @var \Snog\Forms\Entity\SubmitPurchase $submitPurchase */
		$submitPurchase = \XF::em()->find('Snog\Forms:SubmitPurchase', $extraData['purchase_id']);
		if (!$submitPurchase)
		{
			$error = "Unable to find form submit purchase '{$extraData['purchase_id']}'";
			return false;
		}

		return true;
	}

	public function getPurchasableFromExtraData(array $extraData)
	{
		$output = [
			'link' => '',
			'title' => '',
			'purchasable' => null
		];

		/** @var \Snog\Forms\Entity\SubmitPurchase $submitPurchase */
		$submitPurchase = \XF::em()->find('Snog\Forms:SubmitPurchase', $extraData['purchase_id']);
		$form = $submitPurchase->Form;

		if ($submitPurchase && $form)
		{
			$output['link'] = \XF::app()->router('public')->buildLink('form', $form);
			$output['title'] = $form->position;
			$output['purchasable'] = $submitPurchase;
		}

		return $output;
	}

	public function getPurchaseFromExtraData(array $extraData, PaymentProfile $paymentProfile, User $purchaser, &$error = null)
	{
		$purchasable = $this->getPurchasableFromExtraData($extraData);

		/** @var \Snog\Forms\Entity\SubmitPurchase $submitPurchase */
		$submitPurchase = $purchasable['purchasable'];
		if (!$submitPurchase)
		{
			$error = \XF::phrase('this_item_cannot_be_purchased_at_moment');
			return false;
		}

		return $this->getPurchaseObject($paymentProfile, $submitPurchase, $purchaser);
	}

	public function getPurchaseObject(PaymentProfile $paymentProfile, $purchasable, User $purchaser)
	{
		$purchase = new Purchase();

		/** @var \Snog\Forms\Entity\SubmitPurchase $purchasable */

		$form = $purchasable->Form;

		$purchase->title = \XF::phrase('snog_forms_form') . ': ' . $form->position . ' (' . $purchaser->username . ')';
		$purchase->cost = $form->cost_amount + $purchasable->extra_data['extra_cost'] ?? 0;
		$purchase->currency = $form->cost_currency;
		$purchase->purchaser = $purchaser;
		$purchase->paymentProfile = $paymentProfile;
		$purchase->purchasableTypeId = $this->purchasableTypeId;
		$purchase->purchasableId = $purchasable->form_id;
		$purchase->purchasableTitle = $form->position;
		$purchase->extraData = [
			'purchase_id' => $purchasable->purchase_id
		];

		$router = \XF::app()->router('public');

		$returnUrl = $router->buildLink('canonical:form/select', $form, ['purchase_success' => true]);
		$purchase->returnUrl = $form->returnlink ?: $returnUrl;
		$purchase->cancelUrl = $router->buildLink('canonical:form/select', $form);

		return $purchase;
	}

	public function completePurchase(CallbackState $state)
	{
		$paymentResult = $state->paymentResult;
		$purchaseRequest = $state->getPurchaseRequest();

		/** @var \Snog\Forms\Entity\SubmitPurchase $submitPurchase */
		$submitPurchase = \XF::em()->find('Snog\Forms:SubmitPurchase', $purchaseRequest->extra_data['purchase_id']);
		if (!$submitPurchase)
		{
			$state->logType = 'error';
			$state->logMessage = "Form submit purchase not {$purchaseRequest->extra_data['purchase_id']} found.";
			return;
		}

		if ($paymentResult == CallbackState::PAYMENT_RECEIVED)
		{
			$submitPurchase->purchase_request_key = $purchaseRequest->request_key;
			$submitPurchase->purchase_state = 'valid';

			if (!$submitPurchase->save())
			{
				$state->logType = 'error';
				$errors1 = $submitPurchase->getErrors();
				$state->logMessage = reset($errors1);
				return;
			}

			$submitter = $this->setupFormSubmitter($submitPurchase);
			if (!$submitter->validate($errors))
			{
				$state->logType = 'error';
				$state->logMessage = reset($errors);
				return;
			}
			$submitter->save();

			$state->logType = 'payment';
			$state->logMessage = 'Payment received, resource purchased.';
		}
	}

	/**
	 * @param \Snog\Forms\Entity\SubmitPurchase $submitPurchase
	 * @param Thread|null $replyThread
	 * @return \Snog\Forms\Service\Form\Submit
	 */
	protected function setupFormSubmitter(\Snog\Forms\Entity\SubmitPurchase $submitPurchase, ?Thread $replyThread = null)
	{
		// GET QUESTIONS

		/** @var \Snog\Forms\Repository\Question $questionRepo */
		$questionRepo = \XF::repository('Snog\Forms:Question');
		$questions = $questionRepo->getQuestionList($submitPurchase->form_id);

		// GET & SETUP INITIAL VALUES
		$answers = $submitPurchase->extra_data['question'] ?? [];

		/** @var \Snog\Forms\Service\Form\Submit $submitter */
		$submitter = \XF::service('Snog\Forms:Form\Submit', $submitPurchase->Form, $questions);
		$submitter->setIsAutomated();
		$submitter->setAttachmentHash($submitPurchase->extra_data['attachment_hash'] ?? '');
		$submitter->setAnswers($answers);

		if ($replyThread)
		{
			$submitter->setReplyThread($replyThread);
		}

		return $submitter;
	}

	public function reversePurchase(CallbackState $state)
	{
		$purchaseRequest = $state->getPurchaseRequest();

		/** @var \Snog\Forms\Entity\SubmitPurchase $resourcePurchase */
		$resourcePurchase = \XF::em()->findOne(
			'Snog\Forms:SubmitPurchase',
			['purchase_request_key', $purchaseRequest->request_key]
		);

		$resourcePurchase->purchase_state = 'refunded';
		$resourcePurchase->save();

		$state->logType = 'cancel';
		$state->logMessage = 'Payment refunded/reversed.';
	}

	public function getPurchasablesByProfileId($profileId)
	{
		$finder = \XF::finder('Snog\Forms:Form');

		$quotedProfileId = $finder->quote($profileId);
		$uploadColumnName = $finder->columnSqlName('payment_profile_ids');

		$router = \XF::app()->router('admin');
		$forms = $finder->whereSql("FIND_IN_SET($quotedProfileId, $uploadColumnName)")->fetch();
		return $forms->pluck(function (\Snog\Forms\Entity\Form $form, $key) use ($router) {
			return ['snog_forms_form_' . $key, [
				'title' => $this->getTitle() . ': ' . $form->position,
				'link' => $router->buildLink('forms-form/edit', $form)
			]];
		}, false);
	}
}