<?php


namespace Snog\Forms\Repository;


use Snog\Forms\Entity\Form;
use XF\Entity\User;
use XF\Mvc\Entity\Repository;
use XF\PrintableException;

class Log extends Repository
{
	/**
	 * @return \XF\Mvc\Entity\Finder|\Snog\Forms\Finder\Log
	 * @noinspection PhpReturnDocTypeMismatchInspection
	 */
	public function findLogs()
	{
		return $this->finder('Snog\Forms:Log')
			->setDefaultOrder('log_id', 'DESC');
	}

	/**
	 * @return \XF\Mvc\Entity\Finder|\Snog\Forms\Finder\Log
	 * @noinspection PhpReturnDocTypeMismatchInspection
	 */
	public function findLogsByUserForForm(User $user, Form $form, $ipAddress)
	{
		return $this->finder('Snog\Forms:Log')
			->where('user_id', '=', $user->user_id)
			->where('form_id', '=', $form->posid)
			->setDefaultOrder('log_id', 'DESC');
	}

	/**
	 * @param Form $form
	 * @param \Snog\Forms\XF\Entity\User $user
	 * @return int
	 * @throws \XF\PrintableException
	 */
	public function resetCountedFormLogsForUser(Form $form, User $user)
	{
		$updated = $this->db()->update(
			'xf_snog_forms_logs',
			[
				'is_counted' => 0
			],
			'user_id = ? AND form_id = ? AND is_counted = 1',
			[$user->user_id, $form->posid]
		);

		$user->updateAdvancedFormsSerials($form->posid);
		if ($user->isChanged('snog_forms'))
		{
			$user->save(false, false);
		}

		return $updated;
	}

	public function getCooldownStartDate(Form $form, User $user, $ipAddress)
	{
		if ($form->cooldown === 0)
		{
			return 0;
		}

		$logFinder = $this->findLogsByUserForForm($user, $form, $ipAddress)
			->where('is_counted', '=', 1);

		if (!$user->user_id)
		{
			$logFinder->byIp($ipAddress);
		}

		if ($form->cooldown !== -1)
		{
			$logFinder->newerThan(\XF::$time - $form->cooldown);
		}

		$logFinder->orderByDate();

		/** @var \Snog\Forms\Entity\Log $log */
		$log = $logFinder->fetchOne();

		return $log ? $log->log_date : 0;
	}

	/**
	 * @param Form $form
	 * @param User|null $user
	 * @param null $ipAddress
	 * @return \Snog\Forms\Entity\Log
	 * @throws PrintableException
	 */
	public function createLog(
		Form $form,
		User $user = null,
			 $ipAddress = null
	)
	{
		if (!$user)
		{
			$user = \XF::visitor();
		}

		if (!$ipAddress)
		{
			$ipAddress = $this->app()->request()->getIp();
		}

		/** @var \Snog\Forms\Entity\Log $log */
		$log = $this->app()->em()->create('Snog\Forms:Log');
		$log->ip_address = \XF\Util\Ip::convertIpStringToBinary($ipAddress);
		$log->form_id = $form->posid;
		$log->user_id = $user->user_id;

		$log->save();

		return $log;
	}

	public function pruneLogs($cutOff = null)
	{
		if ($cutOff === null)
		{
			$cutOff = \XF::$time - 86400 * 30;
		}

		return $this->db()->delete('xf_snog_forms_logs', 'log_date < ?', $cutOff);
	}

	public function clearLog()
	{
		$this->db()->emptyTable('xf_snog_forms_logs');
	}
}