<?php

namespace Snog\Forms\Repository;

use XF\Html\Renderer\BbCode;
use XF\Mvc\Entity\Repository;

class Question extends Repository
{
	protected $handlerCache = [];

	/**
	 * @Deprecated Temporary solution
	 * @return array
	 */
	public function getSupportedQuestionTypes(): array
	{
		return [
			'text',
			'multiline_text',
			'yes_no',
			'radio',
			'checkboxes',
			'header_phrase',
			'select',
			'multi_select',
			'checkbox',
			'file_upload',
			'date_input',
			'forum_select',
			'wysiwyg',
			'spinbox',
			'thread_prefix',
			'datetime_input',
			'time_input',
			'billable_row',
			'billable_row_total',
		];
	}

	public function getQuestionTypeData()
	{
		$types = $this->getSupportedQuestionTypes();

		$typeData = [];
		foreach ($types as $type)
		{
			$typeData[$type] = [
				'title' => \XF::phrase('snog_forms_question_type_def.' . $type),
				'description' => \XF::phrase('snog_forms_question_type_def_desc.' . $type),
			];
		}

		return $typeData;
	}

	/**
	 * @param int $posid
	 * @return \XF\Mvc\Entity\AbstractCollection|\Snog\Forms\Entity\Question[]
	 * @noinspection PhpReturnDocTypeMismatchInspection
	 */
	public function getQuestionList($posid = 0)
	{
		if ($posid instanceof \Snog\Forms\Entity\Form)
		{
			$posid = $posid->posid;
		}

		return $this->finder('Snog\Forms:Question')
			->order('display')
			->where('posid', '=', $posid)
			->fetch();
	}

	public function createQuestionTree($questions, $rootId = 0)
	{
		return new \XF\Tree($questions, 'display_parent', $rootId);
	}

	public function getQuestionConditionals(\Snog\Forms\Entity\Question $question, $questions, &$uploadNeeded = false)
	{
		$conditionals = [];

		/** @var \Snog\Forms\Entity\Question $conditionQuestion */
		foreach ($questions as $conditionQuestion)
		{
			if (in_array($conditionQuestion->questionid, $question->hasconditional))
			{
				$conditionals[] = [
					'display' => $conditionQuestion->display,
					'answer' => $conditionQuestion->conanswer,
					'questionid' => $conditionQuestion->questionid
				];

				// Detect file upload conditional question type
				if ($conditionQuestion->isUploadNeeded())
				{
					$uploadNeeded = true;
				}
			}
		}

		// Put conditionals in display order
		$arrayColumn = array_column($conditionals, 'answer');
		array_multisort($arrayColumn, SORT_ASC,
			array_column($conditionals, 'display'), SORT_ASC,
			$conditionals);

		return $conditionals;
	}

	public function getFileUploadsAttachReply(\Snog\Forms\Entity\Question $question, $attachments)
	{
		$attachReply = '';

		if ($question->inline > 1)
		{
			$imageFound = false;

			/** @var \XF\Entity\Attachment $attachment */
			foreach ($attachments as $attachment)
			{
				if ($attachment->Data->width > 0)
				{
					if ($question->inline == 2)
					{
						$attachReply .= '[ATTACH=full]' . $attachment->attachment_id . '[/ATTACH]';
					}
					else
					{
						$attachReply .= '[ATTACH]' . $attachment->attachment_id . '[/ATTACH]';
					}

					$imageFound = true;
				}
			}

			if (!$imageFound)
			{
				$attachReply .= \XF::phrase('snog_forms_file_attached');
			}
		}
		else
		{
			$attachReply .= \XF::phrase('snog_forms_file_attached');
		}

		return $attachReply;
	}

	/**
	 * @param \Snog\Forms\Entity\Question $conditionQuestion
	 * @param $attachedFile
	 * @param $questionRepo
	 * @param $attachments
	 * @param $answers
	 * @return array
	 */
	public function getAnswerArray(
		\Snog\Forms\Entity\Question $conditionQuestion,
		$attachedFile,
		$questionRepo,
		$attachments,
		$answers
	): array
	{
		if ($conditionQuestion->type == 'file_upload' && $attachedFile)
		{
			$attachReply = $questionRepo->getFileUploadsAttachReply($conditionQuestion, $attachments);
			$answers[$conditionQuestion->questionid] = $attachReply;
		}

		// PROCESS WYSIWYG EDITOR
		if ($conditionQuestion->type == 'wysiwyg' && !isset($answers[$conditionQuestion->questionid]))
		{
			if (isset($answers[$conditionQuestion->questionid . '_html']))
			{
				$answers[$conditionQuestion->questionid] = BbCode::renderFromHtml($answers[$conditionQuestion->questionid . '_html']);
			}
		}

		return $answers;
	}

	public function getAnswer(
		\Snog\Forms\Entity\Question $question,
		$answers)
	{
		if ($question->type == 'datetime_input')
		{
			$date = $answers[$question->questionid]['date'] ?? null;
			$time = $answers[$question->questionid]['time'] ?? null;
			return "$date $time";
		}
		else
		{
			return $answers[$question->questionid] ?? null;
		}
	}

	public function getConditionAnswer(
		\Snog\Forms\Entity\Question $question,
		\Snog\Forms\Entity\Question $conditionQuestion,
		$answers)
	{
		if ($question->type == 'datetime_input')
		{
			$date = $answers[$conditionQuestion->questionid]['date'] ?? '';
			$time = $answers[$conditionQuestion->questionid]['time'] ?? '';
			return "$date $time";
		}
		else
		{
			return $answers[$conditionQuestion->questionid] ?? null;
		}
	}

	public function deleteQuestions($posId)
	{
		if ($posId instanceof \Snog\Forms\Entity\Form)
		{
			$posId= $posId->posid;
		}

		$db = $this->db();
		$db->delete('xf_snog_forms_questions', 'posid = ?', $posId);
		$db->delete('xf_snog_forms_answers', "posid = ?", $posId);
	}

	public function deleteAllQuestions()
	{
		$db = $this->db();
		$db->emptyTable('xf_snog_forms_questions');
		$db->emptyTable('xf_snog_forms_answers');
	}

	public function getLegacyTypeMap()
	{
		return [
			'text' => 1,
			'multiline_text' => 2,
			'yes_no' => 3,
			'radio' => 4,
			'checkboxes' => 5,
			'header_phrase' => 6,
			'select' => 7,
			'multi_select' => 8,
			'checkbox' => 9,
			'file_upload' => 10,
			'date_input' => 11,
			'forum_select' => 12,
			'wysiwyg' => 13,
			'spinbox' => 14,
			'thread_prefix' => 15,
			'datetime_input' => 16,
			'time_input' => 17
		];
	}
}