<?php

namespace Snog\Forms\Service\Form;

use Snog\Forms\Entity\Form as FormEntity;
use Snog\Forms\Entity\Log;
use Snog\Forms\Entity\Promotion;
use Snog\Forms\Entity\Question;
use XF;
use XF\App;
use XF\Entity\Forum;
use XF\Entity\Thread;
use XF\Entity\User;
use XF\Http\Request;
use XF\Mvc\Entity\ArrayCollection;
use XF\PrintableException;
use XF\Service\AbstractService;
use XF\Service\Thread\Creator;
use XF\Service\Thread\Replier;
use XF\Service\ValidateAndSavableTrait;

class Submit extends AbstractService
{
	use ValidateAndSavableTrait;

	/**
	 * @var FormEntity
	 */
	protected $form;

	protected $performValidations = true;

	/**
	 * @var ArrayCollection|Question[]
	 */
	protected $questions;
	protected $answers;
	protected $unansweredQuestionCount = 0;

	protected $attachmentHash;

	protected $attachmentCount = 0;

	protected $storeAnswers = [];

	protected $formattedAnswers = [];

	protected $reportMessages = [];

	protected $titleAnswers = [];

	protected $answersExtraCost = 0.00;


	/**
	 * @var Thread|null
	 */
	protected $replyThread = null;

	/**
	 * @var Replier|null
	 */
	protected $replier = null;

	/**
	 * @var Creator|null
	 */
	protected $threadCreator = null;

	/**
	 * @var Creator|null
	 */
	protected $secondThreadCreator = null;

	/**
	 * @var |null
	 */
	protected $conversationCreator =  null;

	protected $loggedIp;

	public function __construct(App $app, FormEntity $form, ArrayCollection $questions)
	{
		parent::__construct($app);

		$this->form = $form;
		$this->setQuestions($questions);
		$this->setupDefaults();
	}

	protected function setupDefaults()
	{
		if ($this->form->oldthread)
		{
			$this->replyThread = $this->em()->findOne('XF:Thread', [
				'thread_id', '=', $this->form->oldthread
			], ['Forum', 'Forum.Node']);
		}

		$this->loggedIp = $this->app->request()->getIp();
	}

	protected function setQuestions(ArrayCollection $questions)
	{
		$this->questions = $questions;
	}

	public function setAnswers($answers)
	{
		$this->answers = $answers;
	}

	public function setAttachmentHash($attachmentHash)
	{
		$this->attachmentHash = $attachmentHash;
	}

	public function setReplyThread(Thread $thread)
	{
		$this->replyThread = $thread;
	}

	public function setIsAutomated($isAutomated = true)
	{
		$this->setPerformValidations($isAutomated);
	}

	public function setPerformValidations($performValidations = true)
	{
		$this->performValidations = $performValidations;
	}

	public function getAttachmentCount()
	{
		return $this->attachmentCount;
	}

	public function getUnansweredQuestionCount()
	{
		return $this->unansweredQuestionCount;
	}

	public function getStoreAnswers()
	{
		return $this->storeAnswers;
	}

	public function getThreadCreator()
	{
		return $this->threadCreator;
	}

	public function getSecondThreadCreator()
	{
		return $this->secondThreadCreator;
	}

	public function getReplier()
	{
		return $this->replier;
	}

	public function setLoggedIp($loggedIp)
	{
		$this->loggedIp = $loggedIp;
	}

	public function getAnswersExtraCost()
	{
		return $this->answersExtraCost;
	}

	protected function _validate()
	{
		$user = XF::visitor();

		$answers = $this->answers;
		$form = $this->form;
		$errors = [];

		$attachments = [];
		$attachedFile = false;

		// CHECK IF FILE(S) ARE UPLOADED
		if (!empty($this->attachmentHash))
		{
			$attachRepo = $this->getAttachmentRepo();

			/** @var ArrayCollection|\XF\Entity\Attachment[] $attachments */
			$attachments = $attachRepo->findAttachmentsByTempHash($this->attachmentHash)->fetch();
			$this->attachmentCount = $attachments->count();
			if ($this->performValidations && $form->minimum_attachments && ($this->attachmentCount < $form->minimum_attachments))
			{
				$errors[] = XF::phrase('snog_forms_minimum_attachment_error', ['attachments' => $form->minimum_attachments]);
				return $errors;
			}

			if ($this->attachmentCount)
			{
				$attachedFile = true;
			}
		}

		$questions = $this->questions;

		$questionRepo = $this->getQuestionRepo();
		$hasForumSelectQuestion = false;

		$conditionQuestions = [];

		// GET CONDITIONAL QUESTIONS
		foreach ($questions as $question)
		{
			if ($question->hasconditional)
			{
				$conditionals = $questionRepo->getQuestionConditionals($question, $questions);
				$conditionQuestions[$question->questionid] = $conditionals;
			}

			if ($question->type == 'forum_select')
			{
				$hasForumSelectQuestion = true;
			}
		}

		$forceNodeId = $this->app->request()->filter('node_id', 'uint');
		if ($forceNodeId && !$hasForumSelectQuestion)
		{
			$form->setOption('report_node_id', $forceNodeId);
		}

		// CHECK REQUIRED QUESTIONS
		foreach ($questions as $question)
		{
			// PROCESS MAIN QUESTIONS
			if ($question->conditional)
			{
				// CHECK ANSWERED CONDITIONAL OF CONDITIONAL FOR REQUIRED ANSWERS
				if (isset($answers[$question->questionid]))
				{
					$answers = $this->processConditionalQuestionsAnswers(
						$question,
						$questions,
						$conditionQuestions,
						$answers,
						$attachedFile,
						$attachments,
						$errors
					);
				}

				continue;
			}

			// HANDLE FILE UPLOADS FIRST
			$answers = $questionRepo->getAnswerArray($question, $attachedFile, $questionRepo, $attachments, $answers);

			$answer = $questionRepo->getAnswer($question, $answers);

			// Now check for required questions missing answers
			$errors += $question->getAnswerErrors($answer);

			// PROCESS REQUIRED CONDITIONAL QUESTIONS FOR THIS QUESTION
			$answers = $this->processConditionalQuestionsAnswers(
				$question,
				$questions,
				$conditionQuestions,
				$answers,
				$attachedFile,
				$attachments,
				$errors
			);

			$form->setOption('answers', $answers);
		}

		$isFirstQuestion = true;

		// POSSIBLE FUTURE USE
		//$message .= $form->aboveapp;

		$initialReportMessage = $this->getInitialReportMessage();
		if ($form->incname)
		{
			$isFirstQuestion = false;
		}

		/** @var Question $question */
		foreach ($questions as $question)
		{
			// Process main questions
			if ($question->conditional)
			{
				continue;
			}

			$answer = $questionRepo->getAnswer($question, $answers);

			if ((empty($answer)) && !$question->showunanswered)
			{
				$this->unansweredQuestionCount++;
				continue;
			}

			if ($question->hasAnswerMessage() && $isFirstQuestion)
			{
				$isFirstQuestion = false;
			}

			$this->processAnswer(
				$question,
				$answer,
				$isFirstQuestion
			);

			// SEPARATE ANSWER VALUES TO KEEP BB CODES OUT OF TITLE & DATABASE

			// PROCESS CONDITIONAL QUESTIONS FOR THIS QUESTION
			if (!empty($question->hasconditional) && $answer)
			{
				$this->processConditionals(
					$question,
					$answers,
					$conditionQuestions
				);
			}
		}

		// FINAL MESSAGE ERROR TRAPS
		if ($this->performValidations && $this->unansweredQuestionCount == $questions->count())
		{
			$errors[] = XF::phrase('snog_forms_error_none_answered');
			return $errors;
		}

		if ($this->replyThread)
		{
			$postReportMessage = $initialReportMessage . implode('', $this->reportMessages['post'] ?? []);
			$this->replier = $this->setupReportReplyCreate($this->replyThread, $postReportMessage);
			if (!$this->replier->validate($replierErrors))
			{
				$errors += $replierErrors;
				return $errors;
			}
		}

		$title = $form->subject;

		$formRepo = $this->getFormRepo();
		$title = $formRepo->getReportTitle($title, $this->titleAnswers, $user, $unansweredQuestionIds);

		// TITLE NOT COMPLETE - THROW ERROR
		if (!empty($unansweredQuestionIds))
		{
			$errorCount = count($unansweredQuestionIds);
			$errorIDs = implode(',', $unansweredQuestionIds);

			if ($errorCount > 1)
			{
				$errors[] = XF::phrase('snog_forms_error_title_plural', ['questions' => $errorIDs]);
			}
			else
			{
				$errors[] = XF::phrase('snog_forms_error_title_single', ['question' => $errorIDs]);
			}

			return $errors;
		}

		$title = substr($title, 0, 150);
		$postReportMessage = $initialReportMessage . implode('', $this->reportMessages['post'] ?? []);

		// FORM TO NEW THREAD
		$reportForum = $form->getThreadReportNode();
		if ($reportForum)
		{
			$poster = $this->form->getPoster();

			$threadParams = [
				'title' => $title,
				'message' => $postReportMessage,
				'watch' => $form->watchthread && $poster->user_id == $user->user_id,
				'forum_node' => $reportForum->node_id,
				'attachment_hash' => $this->attachmentHash,
				'create_poll' => true
			];

			XF::asVisitor($poster, function () use ($reportForum, $threadParams) {
				return $this->threadCreator = $this->setupReportThreadCreate($reportForum, $threadParams);
			});

			if (!$this->threadCreator->validate($threadCreatorErrors))
			{
				return $errors + $threadCreatorErrors;
			}
		}

		// FORM TO SECOND THREAD
		$secondReportForum = $form->getThreadReportSecondNode();
		if ($secondReportForum)
		{
			$poster = $form->getSecondaryPoster($user);

			$threadParams = [
				'title' => $title,
				'message' => $postReportMessage,
				'watch' => false,
				'forum_node' => $secondReportForum->node_id,
				'attachment_hash' => null,
				'create_poll' => false
			];

			$this->secondThreadCreator = XF::asVisitor($poster, function () use ($secondReportForum, $threadParams) {
				return $this->setupReportThreadCreate($secondReportForum, $threadParams);
			});

			if (!$this->secondThreadCreator->validate($secondThreadCreatorErrors))
			{
				return $errors + $secondThreadCreatorErrors;
			}
		}

		return $errors;
	}

	protected function _save()
	{
		$form = $this->form;

		$answers = $this->answers;

		/** @var \Snog\Forms\XF\Entity\User $user */
		$user = XF::visitor();

		$attachments = [];

		$db = $this->app->db();
		$db->beginTransaction();

		/** @var \Snog\Forms\Repository\Log $logRepo */
		$logRepo = $this->repository('Snog\Forms:Log');
		$log = $logRepo->createLog($form, $user, $this->loggedIp);

		// CHECK IF FILE(S) ARE UPLOADED
		if ($this->attachmentCount > 0)
		{
			$attachRepo = $this->getAttachmentRepo();

			/** @var ArrayCollection|\XF\Entity\Attachment[] $attachments */
			$attachments = $attachRepo->findAttachmentsByTempHash($this->attachmentHash)->fetch();
		}

		if ($this->storeAnswers)
		{
			/** @var \Snog\Forms\Repository\Answer $answerRepo */
			$answerRepo = $this->repository('Snog\Forms:Answer');
			$answerRepo->saveAnswers($this->storeAnswers, $form->posid, $log->log_id, $user->user_id);
		}

		// POSSIBLE FUTURE USE
		//$message .= $form->belowapp;

		// BUILD REPORT TITLE
		$title = $form->subject;

		$formRepo = $this->getFormRepo();
		$title = $formRepo->getReportTitle($title, $this->titleAnswers, $user, $unansweredQuestionIds);

		$thread = null;

		$initialReportMessage = $this->getInitialReportMessage();

		// FORM TO NEW THREAD
		$reportForum = $form->getThreadReportNode();
		if ($reportForum)
		{
			$threadCreator = $this->threadCreator;
			if ($threadCreator)
			{
				$thread = $threadCreator->save();
				$this->savePollPromotion($thread, $log, $reportForum);
			}
		}

		// FORM TO SECOND THREAD
		$secondReportForum = $form->getThreadReportSecondNode();
		if ($secondReportForum)
		{
			$secondThreadCreator = $this->secondThreadCreator;
			if ($secondThreadCreator)
			{
				$secondThread = $secondThreadCreator->save();
				$this->savePollPromotion($secondThread, $log, $reportForum);
			}
		}

		// FORM TO EXISTING THREAD
		$replier = $this->replier;
		if ($replier)
		{
			/** @var \XF\Entity\Post $returnPost */
			$returnPost = $this->replier->save();
			$this->finalizeReportReplyCreate($this->replier);

			$replyThread = $replier->getThread();

			if ($user->user_id && $form->instant)
			{
				// SAVE PROMOTION INFO FOR APPROVE/DENY

				$promotion = $form->getNewPromotion($user);
				$promotion->post_id = $returnPost->post_id;
				$promotion->thread_id = $replyThread->thread_id;

				$promotion->approve = true;
				$promotion->new_additional = $form->pollpromote;
				$promotion->forum_node = $replyThread->node_id;
				$promotion->log_id = $log->log_id;

				$promotion->save(false, false);
			}

			if ($form->postapproval)
			{
				$returnPost->message_state = 'moderated';
				if ($returnPost->isChanged('message_state'))
				{
					$returnPost->save(false, false);
				}
			}
		}

		$reportMessages = $this->reportMessages;

		// SEND FORM BY PC
		$sender = $form->ConversationUser;
		if ($sender && $form->bypm)
		{
			$conversationReportMessage = $initialReportMessage . implode('', $reportMessages['conversation_message'] ?? []);
			$this->sendReportConversationMessage(
				$title,
				$conversationReportMessage,
				$sender,
				$form->pmto,
				[
					'conversationOptions' => [
						'conversation_open' => !$form->pmdelete,
					],
					'formatMessage' => $form->parseyesno
				]
			);
		}

		// SEND FORM BY EMAIL
		if ($form->email)
		{
			$emailReportMessage = $initialReportMessage . implode('', $reportMessages['email'] ?? []);
			$this->sendReportEmails(
				$title,
				$emailReportMessage,
				$thread,
				$form,
				$user,
				$this->formattedAnswers['email'],
				$this->titleAnswers,
				$attachments,
				$answers
			);
		}

		if ($form->confirmation_email)
		{
			$emailReportMessage = $initialReportMessage . implode('', $reportMessages['email'] ?? []);

			$this->sendConfirmationEmails(
				$title,
				$emailReportMessage,
				$thread,
				$form,
				$user,
				$this->formattedAnswers['email'],
				$this->titleAnswers,
				$attachments,
				$answers
			);
		}

		if ($user->user_id)
		{
			// SEND PC TO USER
			if ($sender && $form->pmapp && $sender->username !== $user->username)
			{
				$initialReportMessage = str_replace('{1}', $user->username, $form->pmtext);
				$initialReportMessage = str_replace('{2}', $this->app->options()->boardTitle, $initialReportMessage);
				$initialReportMessage = str_replace('{3}', $form->pmsender, $initialReportMessage);
				$this->sendReportConversationMessage(
					$title,
					$initialReportMessage,
					$sender,
					$user->username,
					[
						'conversationOptions' => [
							'conversation_open' => !$form->pmdelete,
						],
						'formatMessage' => $form->parseyesno,
					]
				);
			}

			// INSTANT PROMOTE
			if ($form->appadd)
			{
				/** @var \XF\Service\User\UserGroupChange $userGroupService */
				$userGroupService = $this->service('XF:User\UserGroupChange');
				$userGroupService->addUserGroupChange($user->user_id, 'formsInstantPromote' . $form->posid, $form->addto);
			}

			if ($form->apppromote)
			{
				/** @var \Snog\Forms\Repository\Promotion $promotionRepo */
				$promotionRepo = $this->repository('Snog\Forms:Promotion');
				$promotionRepo->applyPrimaryGroupChange($user->user_id, $form->promoteto);
			}

			// INCREMENT USER'S COUNT FOR THIS FORM
			if ($form->formlimit)
			{
				$user->adjustAdvancedFormsSubmitCount($form, 1);
				if ($user->isChanged('snog_forms'))
				{
					$user->save(false, false);
				}
			}
		}

		$form->fastUpdate('submit_count', $form->submit_count + 1);

		$db->commit();

		return $form;
	}


	/**
	 * @param $title
	 * @param string $emailReportMessage
	 * @param Thread|null $thread
	 * @param FormEntity $form
	 * @param $user
	 * @param $email
	 * @param array $titleAnswers
	 * @param $attachments
	 * @param $answers
	 * @return void
	 */
	protected function sendReportEmails(
		$title,
		string $emailReportMessage,
		?Thread $thread,
		FormEntity $form,
		$user,
		$email,
		array $titleAnswers,
		$attachments,
		$answers
	)
	{
		$mail = $this->setupMail($title, $emailReportMessage, $thread ?? null, [
			'form' => $form,
			'reportSender' => $user,
			'formattedAnswers' => $email ?? [],
			'titleAnswers' => $titleAnswers,
			'attachments' => $attachments
		]);

		$toAddresses = $form->getEmailsToReport($answers);
		foreach ($toAddresses as $toEmail)
		{
			$mail->setTo($toEmail);
			$mail->send();
		}
	}

	public function sendConfirmationEmails(
		$title,
		string $emailReportMessage,
		?Thread $thread,
		FormEntity $form,
		$user,
		$email,
		array $titleAnswers,
		$attachments,
		$answers
	)
	{
		$toAddresses = $form->getEmailsForConfirmation($answers);
		if (!$toAddresses)
		{
			return;
		}

		if ($form->confirmation_email_title)
		{
			$title = strtr($form->confirmation_email_title, [
				'{ip}' => $this->loggedIp,
				'{title}' => $form->position
			]);
		}
		else
		{
			$title = XF::phrase('snog_forms_form_submission_confirmation_subject', [
				'title' => $title
			]);
		}

		if ($form->confirmation_email_text)
		{
			$confirmationMessage = strtr($form->confirmation_email_text, [
				'{reportMessage}' => $emailReportMessage,
				'{ip}' => $this->loggedIp,
				'{title}' => $form->position
			]);
		}
		else
		{
			$confirmationMessage = $emailReportMessage;
		}

		$mail = $this->setupMail($title, $confirmationMessage, $thread ?? null, [
			'form' => $form,
			'reportSender' => $user,
			'formattedAnswers' => $email ?? [],
			'titleAnswers' => $titleAnswers,
			'attachments' => $attachments
		]);

		foreach ($toAddresses as $toEmail)
		{
			$mail->setTo($toEmail);
			$mail->send();
		}
	}

	protected function setupMail($title, $message, $thread = null, $extraData = [])
	{
		$mail = $this->app->mailer()->newMail();
		$mail->setSender($this->app->options()->contactEmailAddress);
		$mail->setTemplate('snog_forms_email', [
			'subject' => $title,
			'message' => $message,
			'thread' => $thread ?? null
		]);

		if (!empty($extraData['form']['email_attachments'])
			&& !empty($extraData['attachments']))
		{
			/** @var \XF\Entity\Attachment $attachment */
			foreach ($extraData['attachments'] as $attachment)
			{
				$attachmentData = $attachment->Data;
				$abstractedPath = $attachmentData->getAbstractedDataPath();
				$tempFile = \XF\Util\File::copyAbstractedPathToTempFile($abstractedPath);

				$fileInfo = finfo_open(FILEINFO_MIME_TYPE);
				$contentType = finfo_file($fileInfo, $tempFile);

				if (XF::$versionId >= 2030031)
				{
					$emailAttachment = new \Symfony\Component\Mime\Part\DataPart(fopen($tempFile, 'r'), $attachment->getFilename(), $contentType);
					$mail->getEmailObject()->attachPart($emailAttachment);
				}
				else
				{
					$emailAttachment = (new \Swift_Attachment());
					$emailAttachment->setFilename($attachment->getFilename())
						->setContentType($contentType)
						->setBody(file_get_contents($tempFile), $contentType);

					$mail->getMessageObject()->attach($emailAttachment);
				}
			}
		}

		return $mail;
	}

	protected function processConditionalQuestionsAnswers(
		Question $question,
				 $questions,
				 $conditionQuestions,
				 $answers,
				 $attachedFile,
				 $attachments,
		array    &$errors
	)
	{
		if (!$question->hasconditional)
		{
			return $answers;
		}

		if (!isset($conditionQuestions[$question->questionid]))
		{
			return $answers;
		}

		$questionRepo = $this->getQuestionRepo();

		foreach ($conditionQuestions[$question->questionid] as $condition)
		{
			$match = false;

			$answer = $questionRepo->getConditionAnswer($question, $question, $answers);
			if (!$answer)
			{
				continue;
			}

			// ACCOUNT FOR CHECKBOX ARRAY
			if (is_array($answer) && in_array($condition['answer'], $answer))
			{
				$match = true;
			}
			elseif ($condition['answer'] == $answer)
			{
				$match = true;
			}

			if (!$match && !in_array($question->type, ['date_input', 'datetime_input', 'time_input']))
			{
				continue;
			}

			/** @var Question $conditionQuestion */
			foreach ($questions as $conditionQuestion)
			{
				if ($conditionQuestion->questionid != $condition['questionid'])
				{
					continue;
				}

				// HANDLE FILE UPLOADS FIRST
				$answers = $questionRepo->getAnswerArray($conditionQuestion, $attachedFile, $questionRepo, $attachments, $answers);
				$conditionalAnswer = $questionRepo->getConditionAnswer($question, $conditionQuestion, $answers);

				// Now check for required questions missing answers
				$errors += $conditionQuestion->getAnswerErrors($conditionalAnswer, $answer);
			}
		}

		return $answers;
	}

	protected function processConditionals(
		Question $question,
				 $answers,
				 $conditionQuestions
	)
	{
		$answer = $answers[$question->questionid] ?? null;
		if (!$question->hasconditional || !$answer || !isset($conditionQuestions[$question->questionid]))
		{
			return [];
		}

		$questionRepo = $this->getQuestionRepo();
		foreach ($conditionQuestions[$question->questionid] as $condition)
		{
			/** @var Question $conditionQuestion */
			foreach ($this->questions as $conditionQuestion)
			{
				if ($conditionQuestion->questionid != $condition['questionid'])
				{
					continue;
				}

				// MATCH SINGLE SELECT ANSWER OR ANSWER IN CHECKBOX MULTIPLE ANSWER ARRAY
				if ($question->isValidConditionalAnswer($condition['answer'], $answer))
				{

					$conditionAnswer = $questionRepo->getConditionAnswer($question, $conditionQuestion, $answers);
					if (empty($conditionAnswer) && !$conditionQuestion->showunanswered)
					{
						$this->unansweredQuestionCount++;
						continue;
					}

					$this->processAnswer(
						$conditionQuestion,
						$conditionAnswer,
						false,
					);

					if ($conditionQuestion->hasconditional && $conditionAnswer !== null)
					{
						$this->processConditionals(
							$conditionQuestion,
							$answers,
							$conditionQuestions
						);
					}
				}
			}
		}
	}

	protected function processAnswer(
		Question $question,
				 $answer,
				 $isFirstQuestion
	)
	{
		if ($question->isUnanswered())
		{
			$this->unansweredQuestionCount++;
		}

		$question->setOption('is_first', $isFirstQuestion);

		// STORE ANSWER TO DATABASE IF NOT A HEADER OR FILE UPLOAD
		$form = $this->form;
		if ($form->store && $question->isAnswerStored())
		{
			$this->storeAnswers[] = [
				'questionid' => $question->questionid,
				'answer' => $question->getFormattedAnswer($answer, 'store'),
			];
		}

		if ($question->type == 'forum_select')
		{
			if ($form->inthread && !$form->node_id)
			{
				$form->setOption('report_node_id', $answer);
			}
			elseif ($form->insecthread && !$form->secnode_id && $answer != $form->getOption('report_node_id'))
			{
				$form->setOption('report_second_node_id', $answer);
			}
		}

		if ($question->type == 'thread_prefix' && $answer)
		{
			$threadPrefixes = $answer;
			$form->setOption('report_prefix_ids', $threadPrefixes);
		}

		if ($this->form->isPurchasable())
		{
			$this->answersExtraCost += $question->getFormSubmitExtraCost($answer);
		}

		$this->titleAnswers[$question->display] = $question->getTitleAnswer($answer);

		$reportContentTypes = $form->getReportContentTypes();
		foreach ($reportContentTypes as $reportType)
		{
			$this->formattedAnswers[$reportType][$question->display] = $question->getFormattedAnswer($answer, $reportType);
			$this->reportMessages[$reportType][] = $question->getFormattedReportMessage($answer, $reportType);
		}
	}

	protected function setupReportConversationCreate(
		User $sender,
			 $options
	)
	{
		/** @var \XF\Service\Conversation\Creator $creator */
		$creator = $this->service('XF:Conversation\Creator', $sender);
		$creator->setOptions($options);
		$creator->setIsAutomated();

		return $creator;
	}

	protected function finalizeReportConversationCreate(\XF\Service\Conversation\Creator $creator)
	{
	}

	/**
	 * @param $title
	 * @param $message
	 * @param User $sender
	 * @param $receiver
	 * @param array $options
	 * @return bool
	 * @throws PrintableException
	 */
	protected function sendReportConversationMessage(
		$title,
		$message,
		User $sender,
		$receiver,
		array $options = []
	)
	{
		$options = array_replace([
			'conversationOptions' => [
				'open_invite' => false,
				'conversation_open' => true,
			],
			'formatMessage' => true,
		], $options);

		$creator = $this->setupReportConversationCreate(
			$sender,
			$options['conversationOptions']
		);

		$creator->setRecipients($receiver, false, false);
		$creator->setContent($title, $message, $options['formatMessage']);

		if ($this->attachmentHash)
		{
			$creator->setAttachmentHash($this->attachmentHash);
		}

		if (!$creator->validate($errors))
		{
			\XF::logError(reset($errors));
			return false;
		}

		/** @var \XF\Entity\ConversationMaster $conversation */
		$conversation = $creator->save();
		$this->finalizeReportConversationCreate($creator);

		// DELETE PC FROM SENDER'S INBOX
		if (!$options['conversationOptions']['conversation_open'])
		{
			/** @var \XF\Finder\ConversationUser $finder */
			$finder = $this->finder('XF:ConversationUser');

			/** @var \XF\Entity\ConversationUser $conversation */
			$conversation = $finder->forUser($sender, false)
				->where('conversation_id', $conversation->conversation_id)
				->with(['Recipient'])
				->fetchOne();

			if ($conversation)
			{
				$recipientState = 'deleted_ignored';
				$recipient = $conversation->Recipient;

				if ($recipient)
				{
					$recipient->recipient_state = $recipientState;
					$recipient->save();
				}
			}
		}

		return true;
	}

	/**
	 * @param Forum $forum
	 * @param array $params
	 * @return Creator
	 * @throws \Exception
	 */
	protected function setupReportThreadCreate(
		Forum $forum,
			  $params = []
	)
	{
		$form = $this->form;

		if (!isset($params['title']) || !isset($params['message']))
		{
			throw new \LogicException("Params not defined for thread creation");
		}

		$format = $form->parseyesno;

		/** @var Creator $creator */
		$creator = XF::service('XF:Thread\Creator', $forum);
		$creator->setContent($params['title'], $params['message'], $format);
		$creator->setIsAutomated();

		$attachmentHash = $params['attachment_hash'] ?? null;

		if ($attachmentHash)
		{
			$creator->setAttachmentHash($attachmentHash);
		}

		if ($form->postapproval)
		{
			$creator->setDiscussionState('moderated');
		}
		else
		{
			$creator->setDiscussionState('visible');
		}

		$threadPrefixes = $form->getOption('report_prefix_ids');

		$addOns = $this->app->container('addon.cache');
		$isMultiPrefix = isset($addOns['SV/MultiPrefix']);

		/**
		 * @return array|int|mixed
		 * @var \SV\MultiPrefix\XF\Entity\Forum|Forum $forum
		 */
		$filterUsablePrefixes = function ($prefixIds) use ($forum) {
			if (!is_array($prefixIds))
			{
				return $forum->isPrefixUsable($prefixIds) ? $prefixIds : 0;
			}

			foreach ($prefixIds as $key => $prefixId)
			{
				if (!$forum->isPrefixUsable($prefixId))
				{
					unset($prefixIds[$key]);
				}
			}
			return $prefixIds;
		};

		if ($isMultiPrefix)
		{
			$usablePrefixIds = $filterUsablePrefixes($form->prefix_ids);
		}
		else
		{
			$firstPrefix = array_values($form->prefix_ids)[0] ?? 0;
			$usablePrefixIds = $forum->isPrefixUsable($firstPrefix) ? $firstPrefix : [];
		}

		// SET ADMIN SET THREAD PREFIX
		if (!$threadPrefixes && $usablePrefixIds)
		{
			$creator->setPrefix($usablePrefixIds);
		}
		else
		{
			// USE PREFIX FROM FORM
			$threadPrefixes = $filterUsablePrefixes($threadPrefixes);
			if ($threadPrefixes)
			{
				$creator->setPrefix($threadPrefixes);
			}
			else
			{
				// Use default prefix for forum
				$defaultPrefix = $isMultiPrefix ? $forum->sv_default_prefix_ids : $forum->default_prefix_id;
				if ($defaultPrefix)
				{
					$creator->setPrefix($defaultPrefix);
				}
			}
		}

		$createPoll = $params['create_poll'] ?? null;
		if ($createPoll && ($form->normalpoll || $form->postpoll))
		{
			/** @var \Snog\Forms\ControllerPlugin\Poll $pollPlugin */
			$pollPlugin = $this->app->controller('Snog\Forms:Form', $this->app->request())
				->plugin('Snog\Forms:Poll');

			if ($forum->isThreadTypeCreatable('poll'))
			{
				$creator->setDiscussionTypeAndData('poll', new Request($this->app->inputFilterer(), []));
				$typeDataSaver = $creator->getTypeDataSaver();

				if ($typeDataSaver instanceof \XF\Service\Thread\TypeData\PollCreator)
				{
					$pollCreatorSvc = $typeDataSaver->getPollCreator();

					// CREATE PROMOTION POLL
					if (!$form->normalpoll && $form->postpoll && $form->pollquestion)
					{
						$pollPlugin->setupPollCreatorSvc($pollCreatorSvc, $form, 1);
					}

					// CREATE NORMAL POLL
					if ($form->normalpoll && !$form->postpoll && $form->normalquestion)
					{
						$pollPlugin->setupPollCreatorSvc($pollCreatorSvc, $form, 2);
					}
				}
			}
			else
			{
				\XF::logError("Advanced Forms: Poll is not creatable for form '{$form->appid}' in node '{$forum->node_id}'");
			}
		}

		return $creator;
	}

	protected function finalizeReportThreadCreate(Creator $creator, $params = [])
	{
		$creator->sendNotifications();

		$visitor = XF::visitor();
		if ($visitor->user_id)
		{
			$thread = $creator->getThread();

			/** @var \XF\Repository\Thread $threadRepo */
			$threadRepo = $this->repository('XF:Thread');
			$threadRepo->markThreadReadByVisitor($thread, $thread->post_date);

			if ($thread->discussion_state == 'moderated')
			{
				$this->app->session()->setHasContentPendingApproval();
			}

			if ($thread && isset($params['watch']) && $params['watch'])
			{
				$state = $visitor->Option->creation_watch_state == 'watch_email' ? 'watch_email' : 'watch_no_email';

				/** @var \XF\Repository\ThreadWatch $threadWatchRepo */
				$threadWatchRepo = $this->repository('XF:ThreadWatch');
				$threadWatchRepo->setWatchState($thread, $visitor, $state);
			}
		}
	}

	/**
	 * @param Forum $forum
	 * @param array $params
	 * @return Thread
	 * @throws \Exception
	 */
	protected function createReportThread(
		Forum $forum,
			  $params = []
	)
	{
		if (empty($params))
		{
			throw new \LogicException("Params not defined for thread creation");
		}

		$creator = $this->setupReportThreadCreate($forum, $params);
		if (isset($params['create_poll']))
		{
			$this->applyPollToThreadCreator($forum, $creator);
		}

		if (!$creator->validate($errors))
		{
			\XF::logError(reset($errors));
			return null;
		}

		/** @var Thread $thread */
		$thread = $creator->save();
		$this->finalizeReportThreadCreate($creator, $params);

		return $thread;
	}


	/**
	 * @param Thread $thread
	 * @param $message
	 * @return Replier
	 * @throws \Exception
	 */
	protected function setupReportReplyCreate(
		Thread $thread,
			   $message
	)
	{
		$poster = $this->form->getPoster();

		return XF::asVisitor($poster, function () use ($thread, $message) {
			/** @var Replier $replier */
			$replier = XF::service('XF:Thread\Replier', $thread);
			$replier->setIsAutomated();
			$replier->setMessage($message, $this->form->parseyesno);

			if ($this->attachmentHash)
			{
				$replier->setAttachmentHash($this->attachmentHash);
			}

			return $replier;
		});
	}

	protected function finalizeReportReplyCreate(Replier $replier)
	{
		$replier->sendNotifications();
	}

	/**
	 * @param Thread $thread
	 * @param Log $log
	 * @param Forum $forum
	 * @return Promotion|null
	 * @throws PrintableException
	 */
	protected function savePollPromotion(Thread $thread, Log $log, Forum $forum)
	{
		$user = \XF::visitor();
		$form = $this->form;

		/** @var \XF\Entity\Poll $poll */
		if ($thread->discussion_type == 'poll')
		{
			$poll = $thread->Poll;
		}

		// SET UP FOR APPROVE/DENY LINKS AND SAVE PROMOTION INFO
		if ($user->user_id && ($form->postpoll || $form->instant))
		{
			// SAVE PROMOTION INFO FOR APPROVE/DENY AND POLL RESULT

			$promotion = $form->getNewPromotion($user);
			$promotion->post_id = $thread->first_post_id;
			$promotion->thread_id = $thread->thread_id;
			if (isset($poll->poll_id) && $form->postpoll)
			{
				$promotion->poll_id = $poll->poll_id;
			}

			if (isset($poll->close_date) && $form->postpoll)
			{
				$promotion->close_date = $poll->close_date;
			}

			if ($form->instant)
			{
				$promotion->approve = true;
			}

			$promotion->forum_node = $forum->node_id;
			$promotion->log_id = $log->log_id;

			$promotion->save(false, false);

			return $promotion;
		}

		return null;
	}

	/**
	 * @return string
	 */
	protected function getInitialReportMessage(): string
	{
		$initialReportMessage = '';

		$form = $this->form;

		// INCLUDE NAME OR IP ADDRESS
		if ($form->incname)
		{
			$user = \XF::visitor();
			if ($user->username)
			{
				$usernameQuestion = '[B]' . XF::phrase('user_name') . ':[/B] ';
				$initialReportMessage .= $form->getWrappedQuestionMessage($usernameQuestion);
				$initialReportMessage .= $form->getWrappedAnswerMessage($user->username);
			}
			else
			{
				$ipAddressQuestion = '[B]' . XF::phrase('ip_address') . ':[/B] ';
				$initialReportMessage .= $form->getWrappedQuestionMessage($ipAddressQuestion);
				$initialReportMessage .= $form->getWrappedAnswerMessage($this->loggedIp);

				if ($form->bbstart && $form->bbend)
				{
					$initialReportMessage = $form->bbstart . $initialReportMessage . $form->bbend;
				}
			}
		}

		return $initialReportMessage;
	}

	/**
	 * @param Forum $forum
	 * @param Creator $creator
	 * @return void
	 */
	protected function applyPollToThreadCreator(Forum $forum, Creator $creator): void
	{
		$form = $this->form;

		/** @var \Snog\Forms\ControllerPlugin\Poll $pollPlugin */
		$pollPlugin = $this->app->controller('XF:Index', $this->app->request())->plugin('Snog\Forms:Poll');

		if (($form->normalpoll || $form->postpoll))
		{
			if (!$forum->isThreadTypeCreatable('poll'))
			{
				throw new \LogicException("Poll is not creatable in forum '{$forum->title}' for form '{$form->position}'. Please contact adminstrators.");
			}

			$creator->setDiscussionTypeAndData('poll', new Request($this->app->inputFilterer(), []));
			$typeDataSaver = $creator->getTypeDataSaver();

			if ($typeDataSaver instanceof \XF\Service\Thread\TypeData\PollCreator)
			{
				$pollCreatorSvc = $typeDataSaver->getPollCreator();

				// CREATE PROMOTION POLL
				if (!$form->normalpoll && $form->postpoll && $form->pollquestion)
				{
					$pollPlugin->setupPollCreatorSvc($pollCreatorSvc, $form, 1);
				}
				elseif (!$form->postpoll && $form->normalpoll && $form->normalquestion)
				{
					// CREATE NORMAL POLL
					$pollPlugin->setupPollCreatorSvc($pollCreatorSvc, $form, 2);
				}
			}
		}
	}

	/**
	 * @return \XF\Repository\Attachment|\XF\Mvc\Entity\Repository
	 */
	protected function getAttachmentRepo()
	{
		return $this->repository('XF:Attachment');
	}

	/**
	 * @return \Snog\Forms\Repository\Question|\XF\Mvc\Entity\Repository
	 */
	protected function getQuestionRepo()
	{
		return $this->repository('Snog\Forms:Question');
	}

	/**
	 * @return \Snog\Forms\Repository\Form|\XF\Mvc\Entity\Repository
	 */
	protected function getFormRepo()
	{
		return $this->repository('Snog\Forms:Form');
	}

}