<?php


namespace Snog\Forms\Service;


use XF\Mvc\Entity\Entity;

class Import extends \XF\Service\AbstractService
{
	protected $newPosIds = [];
	protected $newAppIds = [];
	protected $newQuestionIds = [];

	protected $doMerge = false;

	protected $db;

	public function __construct(\XF\App $app)
	{
		parent::__construct($app);

		$this->db = $app->db();
	}

	public function setDoMerge($merge)
	{
		$this->doMerge = $merge;
	}

	public function importXml($xml)
	{
		if (function_exists('set_time_limit')) @set_time_limit(0);

		if (!$this->doMerge)
		{
			$this->deleteAllImportData();
		}

		foreach ($xml->children() as $child)
		{
			$levelName = $child->getName();

			if ($levelName == 'Type')
			{
				$importData = $this->getTypeImportData($child->children(), $oldId);
				$this->db->insert('xf_snog_forms_types', $importData);

				$this->newAppIds[] = ['oldid' => $oldId, 'newid' => $this->db->lastInsertId()];
			}
			elseif ($levelName == 'Form')
			{
				$importData = $this->getFormImportData($child->children(), $oldId);
				$this->db->insert('xf_snog_forms_forms', $importData);

				$newId = $this->db->lastInsertId();
				if ($oldId !== $newId)
				{
					$this->newPosIds[] = ['oldid' => $oldId, 'newid' => $newId];
				}
			}
			elseif ($levelName == 'Question')
			{
				$importData = $this->getQuestionImportData($child->children(), $oldId);
				$this->db->insert('xf_snog_forms_questions', $importData);

				$newId = $this->db->lastInsertId();
				if ($oldId !== $newId)
				{
					$this->newQuestionIds[] = ['oldid' => $oldId, 'newid' => $newId];
				}
			}
		}

		$this->rebuildConditionals();
	}

	public function getTypeImportData($items, &$oldId = 0)
	{
		$jsonArrays = [
			'user_criteria'
		];

		$data = [];
		foreach ($items as $item)
		{
			$name = $item->getName();

			if (!in_array($name, $jsonArrays))
			{
				// SET OLD VALUE OF TYPE APPID
				if ($name == 'appid')
				{
					$oldId = $item;
				}

				if ($name !== 'appid')
				{
					$data[$name] = html_entity_decode($item);
				}
			}
			else
			{
				// CONVERT TO JSON ARRAY
				$tempItem = html_entity_decode($item);
				$tempArray = \XF\Util\Php::safeUnserialize($tempItem);
				$data[$name] = json_encode($tempArray);
			}
		}

		$defaultData = $this->getDefaultColumnDataForEntity('Snog\Forms:Type');
		return array_replace($defaultData, $data);
	}

	protected function getFormImportData($items, &$oldId)
	{
		$data = [];

		$jsonArrays = [
			'addto',
			'pollpromote',
			'user_criteria',
			'response',
			'forummod',
			'supermod',
			'qrforums'
		];

		$oldId = 0;

		foreach ($items as $item)
		{
			$name = $item->getName();

			// SET OLD VALUE OF FORM POSID
			if ($name == 'posid') $oldId = $item;

			if ($name !== 'posid')
			{
				if (!in_array($name, $jsonArrays))
				{
					if ($name == 'appid')
					{
						// USE NEW APPID
						$oldid = $item;
						foreach ($this->newAppIds as $newAppId)
						{
							if (intval($oldid) == intval($newAppId['oldid']))
							{
								$item = $newAppId['newid'];
								break;
							}
						}
					}
					// Remap old
					elseif ($name == 'prefix_id')
					{
						$name = 'prefix_ids';
					}
					elseif ($name == 'email_attachments') // Fixes import errors before 2.2.4
					{
						$data[$name] = intval($item);
						continue;
					}
					elseif ($name == 'confirmation_email') // Fixes import errors before 2.2.6
					{
						$data[$name] = boolval($item);
						continue;
					}

					$data[$name] = html_entity_decode($item);
				}
				else
				{
					// CONVERT TO JSON ARRAY
					$tempItem = html_entity_decode($item);
					$tempArray = \XF\Util\Php::safeUnserialize($tempItem);
					$data[$name] = json_encode($tempArray);
				}

			}
		}

		$defaultData = $this->getDefaultColumnDataForEntity('Snog\Forms:Form');
		return array_replace($defaultData, $data);
	}

	protected function getQuestionImportData($items, &$oldId = 0)
	{
		$data = [];

		$jsonArrays = [
			'hasconditional',
			'type_data'
		];

		$oldId = 0;

		/** @var  \Snog\Forms\Repository\Question $questionRepo */
		$questionRepo = $this->app->repository('Snog\Forms:Question');
		$legacyTypeMap = $questionRepo->getLegacyTypeMap();

		/** @var \SimpleXMLElement $item */
		foreach ($items as $item)
		{
			$name = $item->getName();

			if ($name == 'questionid')
			{
				$oldId = intval($item);
			}
			else
			{
				if (!in_array($name, $jsonArrays))
				{
					if ($name == 'posid')
					{
						// USE NEW POSID
						$oldid = $item;
						foreach ($this->newPosIds as $newPosId)
						{
							if (intval($oldid) == intval($newPosId['oldid']))
							{
								$item = $newPosId['newid'];
								break;
							}
						}
					}

					if ($name == 'type')
					{
						$newType = array_search($item, $legacyTypeMap);
						if ($newType)
						{
							$item = $newType;
						}
					}

					// USE NEW PARENT QUESTION ID
					if ($name == 'conditional')
					{
						foreach ($this->newQuestionIds as $newQid)
						{
							if ($item == $newQid['oldid'])
							{
								$item = $newQid['newid'];
								break;
							}
						}
					}
					elseif ($name == 'readonly') // Fixes import errors before 2.2.4
					{
						$data[$name] = intval($item);
						continue;
					}

					$data[$name] = html_entity_decode($item);
				}
				else
				{
					// CONVERT TO JSON ARRAY
					$tempItem = html_entity_decode($item);
					$tempArray = \XF\Util\Php::safeUnserialize($tempItem);
					$data[$name] = json_encode($tempArray);
				}

				if (empty($data['type_data']))
				{
					$data['type_data'] = '{}';
				}
			}
		}

		$defaultData = $this->getDefaultColumnDataForEntity('Snog\Forms:Question');
		return array_replace($defaultData, $data);
	}

	protected function rebuildConditionals()
	{
		/** @var \Snog\Forms\Entity\Question[] $questions */
		$questions = $this->finder('Snog\Forms:Question')
			->where('hasconditional', 'NOT LIKE', '[]')
			->fetch();

		foreach ($questions as $question)
		{
			$conditionArray = [];

			if (!empty($question->hasconditional))
			{
				/** @var \Snog\Forms\Entity\Question[] $conditionalQuestions */
				$conditionalQuestions = $this->finder('Snog\Forms:Question')
					->where('conditional', $question->questionid)
					->fetch();

				if ($conditionalQuestions)
				{
					foreach ($conditionalQuestions as $conditionalQuestion)
					{
						$conditionArray[] = $conditionalQuestion->questionid;
					}

					$question->hasconditional = $conditionArray;
					$question->save();
				}
			}
		}
	}

	protected function deleteAllImportData()
	{
		/** @var \Snog\Forms\Repository\Type $typeRepo */
		$typeRepo = $this->repository('Snog\Forms:Type');
		$typeRepo->deleteAllTypes();

		/** @var \Snog\Forms\Repository\Form $formRepo */
		$formRepo = $this->repository('Snog\Forms:Form');
		$formRepo->deleteAllForms();

		/** @var \Snog\Forms\Repository\Question $questionRepo */
		$questionRepo = $this->repository('Snog\Forms:Question');
		$questionRepo->deleteAllQuestions();
	}

	/**
	 * Convert default entity values to the correct format for XML exports
	 *
	 * @param $entity
	 * @return array
	 */
	protected function getDefaultColumnDataForEntity($entity): array
	{
		$defaultData = [];
		$structure = $this->app->em()->getEntityStructure($entity);
		foreach ($structure->columns as $key => $column)
		{
			if (isset($column['default']))
			{
				if (is_array($column['default']))
				{
					$defaultData[$key] = json_encode($column['default']);
				}
				elseif ($column['type'] === Entity::BOOL)
				{
					$defaultData[$key] = $column['default'] ? 1 : 0;
				}
				elseif ($column['type'] === Entity::INT || $column['type'] === Entity::UINT)
				{
					$defaultData[$key] = intval($column['default']) ?: 0;
				}
				else
				{
					$defaultData[$key] = strval($column['default']);
				}
			}
		}

		return $defaultData;
	}
}
