<?php
    class CURLRequester
    {
        const METHOD_GET = 'GET';
        const METHOD_POST = 'POST';
        const METHOD_PUT = 'PUT';
        const METHOD_PATCH = 'PATCH';
        const METHOD_DELETE = 'DELETE';

        private $url;
        private $method;
        private $headers;
        private $json;
        private $debug;
        public $code;

        public $http_errors = array(
            400 => '400 Bad Request',
            401 => '401 Unauthorized',
            500 => '500 Internal Server Error',
            501 => '501 Not Implemented',
            502 => '502 Bad Gateway',
            503 => '503 Service Unavailable',
            504 => '504 Gateway Timeout',
            429 => '429 Rate Limit Reached',
        );

        public function __construct($url, $method = 'GET', $json = false, $debug = false)
        {
            $this->url = $url;
            $this->json = $json;
            $this->method = $method;
            $this->debug = $debug;
            $this->headers = array();
            $this->code = 0;
        }

        public function setMethod($method)
        {
            $this->method = $method;
        }

        public function addHeader($name, $value = null)
        {
            if (null == $value) {
                $this->headers[] = $name;
            } else {
                $this->headers[] = $name.': '.$value;
            }
        }

        public function request($data = null, $json = false, $timeout = 20, $isfile = false, $auth = false, $redirects = 0,
                                $cainfo = null, $sslcert = null, $sslkey = null)
        {
            $curl = curl_init();
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($curl, CURLOPT_HEADER, 1);

            $url = $this->url;

            curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, 1);
            curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, 2);
            //curl_setopt($curl, CURLOPT_CAINFO, str_replace('\\', '/', __DIR__).'/../../cacert.pem');
            //curl_setopt($curl, CURLOPT_FOLLOWLOCATION, 1);
            curl_setopt($curl, CURLOPT_VERBOSE, 0);
            curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, $timeout);
            curl_setopt($curl, CURLOPT_TIMEOUT, $timeout);

            if($cainfo != null){
                curl_setopt($curl, CURLOPT_CAINFO, $cainfo);
            }

            if($sslcert != null){
                curl_setopt($curl, CURLOPT_SSLCERT, $sslcert);
            }

            if($sslkey != null){
                curl_setopt($curl, CURLOPT_SSLKEY, $sslkey);
            }


            if ($auth) {
                curl_setopt($curl, CURLOPT_USERPWD, $auth);
            }

            $filename = false;

            if ($isfile) {
                $filename = tempnam(sys_get_temp_dir(), 'curl_file');

                $fp = fopen($filename);

                curl_setopt($curl, CURLOPT_FILE, $fp);
            }

            if (0 !== strcasecmp($this->method, self::METHOD_GET)) {
                if (0 === strcasecmp($this->method, self::METHOD_POST)) {
                    curl_setopt($curl, CURLOPT_POST, true);
                } else {
                    curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $this->method);
                }

                curl_setopt($curl, CURLOPT_POSTFIELDS, '');
            }

            if (!empty($data)) {
                if ($json) {
                    $fields = is_array($data) ? ToJson($data) : (string) $data;
                } else {
                    $fields = is_array($data) ? http_build_query($data) : (string) $data;
                }

                if (self::METHOD_GET == $this->method && !$json) {
                    if (false === strpos($url, '?')) {
                        $url .= '?'.$fields;
                    } else {
                        $url .= '&'.$fields;
                    }
                } else {
                    curl_setopt($curl, CURLOPT_POSTFIELDS, $fields);
                }
            }

            curl_setopt($curl, CURLOPT_URL, $url);

            $add = array();

            if ($json) {
                $add[] = 'Content-Type: application/json';
            }

            if (!empty($this->headers)) {
                curl_setopt($curl, CURLOPT_HTTPHEADER, array_merge($this->headers, $add));
            }

            $response = curl_exec($curl);

            $header_size = curl_getinfo($curl, CURLINFO_HEADER_SIZE);
            $header = substr($response, 0, $header_size);
            $body = substr($response, $header_size);

            if ($this->debug) {
                Debug(ToJson(array('url' => $this->url, 'method' => $this->method, 'request' => $data, 'response-header' => $header, 'response-body' => $body)));
            }

            $code = curl_getinfo($curl, CURLINFO_HTTP_CODE);

            $this->code = $code;

            if (array_key_exists($code, $this->http_errors)) {
                Debug('Response Http Error - '.$code.' ('.$this->http_errors[$code].')');

                return false;
            } elseif (301 == $code || 302 == $code) {
                if ($redirects > 10) {
                    Debug('Unable to connect to '.$this->url.' Error: Too many redirects (10+)');

                    return false;
                }

                list($headers) = explode("\r\n\r\n", $header, 2);

                $matches = array();
                preg_match("/(Location:|URI:)[^(\n)]*/", $headers, $matches);

                if(sizeof($matches) < 2) {
                    Debug('Unable to connect to '.$this->url.' Error: Could not parse redirect.');

                    return false;
                }

                $url = trim(str_replace($matches[1], '', $matches[0]));

                $url_parsed = parse_url($url);
                if (isset($url_parsed)) {
                    $this->url = $url;

                    ++$redirects;

                    return $this->request($data, $json, $timeout, $isfile, $auth, $redirects);
                }
            }

            $code = curl_errno($curl);

            if (0 < $code) {
                Debug('Unable to connect to '.$this->url.' Error: '.$code.': '.curl_error($curl));

                return false;
            }

            curl_close($curl);

            if ($filename) {
                return $filename;
            }

            return $this->json ? FromJson($body) : $body;
        }
    }
