<?php
/**
 * @author
 *
 **/
class Paysafecard
{
    private $response;
    private $request = array();
    private $curl;
    private $key = '';
    private $url = '';
    private $environment = 'TEST';

    public function __construct($key = '', $environment = 'TEST')
    {
        $this->key = $key;
        $this->environment = $environment;
        $this->setEnvironment();
    }

    /**
     * send curl request.
     *
     * @param assoc array  $curlparam
     * @param httpmethod   $method
     * @param string array $header
     */
    private function doRequest($curlparam, $method, $headers = array())
    {
        $ca = null; $cert = null; $key = null;

        if(!empty(Config::Get('paysafecard_cert_path'))){
            $cert = Config::Get('paysafecard_cert_path');
            $key = Config::Get('paysafecard_key_path');
            $ca = Config::Get('paysafecard_ca_path');
        }

        $request = new CURLRequester($this->url, $method, true, true);
        $request->addHeader('Authorization', 'Basic '.base64_encode($this->key));

        foreach ($headers as $header) {
            $request->addHeader($header);
        }

        $this->response = $request->request($curlparam, true, 5, false, false, 0, $ca, $cert, $key);

        $this->setEnvironment();
    }

    /**
     * check request status.
     *
     * @return bool
     */
    public function requestIsOk()
    {
        if (false !== $this->response) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * get the request.
     *
     * @return mixed request
     */
    public function getRequest()
    {
        return $this->request;
    }

    /**
     * get curl.
     *
     * @return mixed curl
     */
    public function getCurl()
    {
        return $this->curl;
    }

    /**
     * create a payment.
     *
     * @param float              $amount
     * @param string             $currency
     * @param string             $customer_id
     * @param string             $customer_ip
     * @param string             $success_url
     * @param string             $failure_url
     * @param string             $notification_url
     * @param string|float       $correlation_id
     * @param string|countrycode $country_restriction
     * @param int                $min_age
     * @param int                $shop_id
     *
     * @return mixed|response
     */
    public function createPayment($amount, $currency, $customer_id, $customer_ip, $success_url, $failure_url, $notification_url, $correlation_id = '', $country_restriction = '', $kyc_restriction = '', $min_age = '', $shop_id = '', $submerchant_id = '')
    {
        $amount = str_replace(',', '.', $amount);

        $customer = array(
            'id' => $customer_id,
            'ip' => $customer_ip,
        );
        if ('' != $country_restriction) {
            array_push($customer, [
                'country_restriction' => $country_restriction,
            ]);
        }

        if ('' != $kyc_restriction) {
            array_push($customer, [
                'kyc_level' => $kyc_restriction,
            ]);
        }

        if ('' != $min_age) {
            array_push($customer, [
                'min_age' => $min_age,
            ]);
        }

        $jsonarray = array(
            'currency' => $currency,
            'amount' => $amount,
            'customer' => $customer,
            'redirect' => array(
                'success_url' => $success_url,
                'failure_url' => $failure_url,
            ),
            'type' => 'PAYSAFECARD',
            'notification_url' => $notification_url,
            'shop_id' => $shop_id,
        );

        if ('' != $submerchant_id) {
            array_push($jsonarray, [
                'submerchant_id' => $submerchant_id,
            ]);
        }

        if ('' != $correlation_id) {
            $headers = ['Correlation-ID: '.$correlation_id];
        } else {
            $headers = [];
        }
        $this->doRequest($jsonarray, 'POST', $headers);
        if (true === $this->requestIsOk()) {
            return $this->response;
        } else {
            return false;
        }
    }

    /**
     * get the payment id.
     *
     * @param string $payment_id
     *
     * @return response|bool
     */
    public function capturePayment($payment_id)
    {
        $this->url = $this->url.$payment_id.'/capture';
        $jsonarray = array(
            'id' => $payment_id,
        );
        $this->doRequest($jsonarray, 'POST');
        if (true == $this->requestIsOk()) {
            return $this->response;
        } else {
            return false;
        }
    }

    /**
     * retrieve a payment.
     *
     * @param string $payment_id
     *
     * @return response|bool
     */
    public function retrievePayment($payment_id)
    {
        $this->url = $this->url.$payment_id;
        $jsonarray = array();
        $this->doRequest($jsonarray, 'GET');
        if (true == $this->requestIsOk()) {
            return $this->response;
        } else {
            return false;
        }
    }

    /**
     * get the response.
     *
     * @return mixed
     */
    public function getResponse()
    {
        return $this->response;
    }

    /**
     * set environmente.
     *
     * @return mixed
     */
    private function setEnvironment()
    {
        if ('TEST' == $this->environment && !tempty(Config::Get('paysafecard_cert_path'))) {
            $this->url = 'https://api-cg.test.paysafecard.com/v1/payments/';
        } elseif ('PRODUCTION' == $this->environment  && !tempty(Config::Get('paysafecard_cert_path'))) {
            $this->url = 'https://api-cg.paysafecard.com/v1/payments/';
        } elseif ('TEST' == $this->environment) {
            $this->url = 'https://apitest.paysafecard.com/v1/payments/';
        } elseif ('PRODUCTION' == $this->environment) {
            $this->url = 'https://api.paysafecard.com/v1/payments/';
        } else {
            echo 'Environment not supported';

            return false;
        }
    }

    /**
     * get error.
     *
     * @return response
     */
    public function getError()
    {
        if (!isset($this->response['number'])) {
            switch ($this->curl['info']['http_code']) {
                case 400:
                    $this->response['number'] = 'HTTP:400';
                    $this->response['message'] = 'Logical error. Please check logs.';
                    break;
                case 403:
                    $this->response['number'] = 'HTTP:403';
                    $this->response['message'] = 'Transaction could not be initiated due to connection problems. The IP from the server is not whitelisted! Server IP:'.$_SERVER['SERVER_ADDR'];
                    break;
                case 500:
                    $this->response['number'] = 'HTTP:500';
                    $this->response['message'] = 'Server error. Please check logs.';
                    break;
            }
        }
        switch ($this->response['number']) {
            case 4003:
                $this->response['message'] = 'The amount for this transaction exceeds the maximum amount. The maximum amount is 1000 EURO (equivalent in other currencies)';
                break;
            case 3001:
                $this->response['message'] = 'Transaction could not be initiated because the account is inactive.';
                break;
            case 2002:
                $this->response['message'] = 'payment id is unknown.';
                break;
            case 2010:
                $this->response['message'] = 'Currency is not supported.';
                break;
            case 2029:
                $this->response['message'] = 'Amount is not valid. Valid amount has to be above 0.';
                break;
            default:
                $this->response['message'] = 'Transaction could not be initiated due to connection problems. If the problem persists, please contact our support. ';
                break;
        }

        return $this->response;
    }

    public static function GetKey(){
        $key = Settings::Get('settings_donations_paysafecard_key');

        if(!tempty(Config::Get('paysafecard_key'))){
            $key = Config::Get('paysafecard_key');
        }

        return $key;
    }

    public static function PostProcess($id, $gex_paymentid, $second_try = false)
    {
        $db = MysqliDb::getInstance();

        $pscpayment = new Paysafecard(Paysafecard::GetKey(), Settings::Get('settings_donations_paysafecard_environment'));

        $response = $pscpayment->retrievePayment($id);

        if (true == $response) {
            if (isset($response['object'])) {
                if ('AUTHORIZED' == $response['status']) {
                    // capture payment
                    $response = $pscpayment->capturePayment($id);

                    if (true == $response) {
                        if (isset($response['object'])) {
                            $payment = Payment::PostProcess($gex_paymentid);

                            if ('SUCCESS' == $response['status']) {
                                $customer_id = $response['customer']['id'];

                                if ($payment) {
                                    Debug('Payment found. Payment Hash: '.$response['customer']['id']);

                                    $buyer = new User($payment['buyer']);
                                    $for = new User($payment['for']);
                                    $coupon = new Coupon($payment['coupon']);
                                    $price = $payment['price'];
                                    $package = new Package($payment['package'], null, true, $buyer);

                                    if ($buyer->valid && $for->valid && $package->valid && $price) {
                                        if ($customer_id == md5($buyer->GetValue('steamid64'))) {
                                            if (Payment::Validate('paysafecard', array($response, $package, $price))) {
                                                Debug('Payment verified.');

                                                $amount = $response['amount'];

                                                $db->where('transactionid', $id);
                                                $db->getOne('donations');

                                                if ($db->count) {
                                                    Debug('TransactionID already proceeded');

                                                    return true;
                                                }

                                                if ($payment['coupon']) {
                                                    if ($coupon->valid) {
                                                        $coupon->Devaluate();
                                                    } else {
                                                        Payment::CouponRefund($for, $id, $amount);
                                                        Debug('Invalid coupon. Refunding...');

                                                        return false;
                                                    }
                                                }

                                                $data = array(
                                                    'name_buyer' => $buyer->GetValue('nick'),
                                                    'steamid64_buyer' => $buyer->GetValue('steamid64'),
                                                    'steamid64_for' => $for->GetValue('steamid64'),
                                                    'amount' => $amount,
                                                    'package' => $package->GetValue('id'),
                                                    'transactionid' => $id,
                                                    'coupon' => $payment['coupon'],
                                                    'email' => '',
                                                    'gateway' => 'paysafecard',
                                                );

                                                Payment::Finish($data, $buyer, $for, $package, $payment);

                                                return true;
                                            }
                                        }
                                    }
                                }
                            } else {
                                if (2017 == $error['number']) {
                                    $response = $pscpayment->retrievePayment($id);

                                    if (isset($response['status'])) {
                                        if ('SUCCESS' == $response['status']) {
                                            return true;
                                        } else {
                                            return false;
                                        }
                                    } else {
                                        return false;
                                    }
                                }
                            }
                        }
                    }

                    if (!$second_try) {
                        return Paysafecard::PostProcess($id, $gex_paymentid, true);
                    } else {
                        Debug('Paysafecard payment could not be post-processed. status: AUTHORIZED');
                    }
                } elseif ('SUCCESS' == $response['status']) {
                    return true;
                }
            }
        }

        Debug('Paysafecard payment could not be post-processed.');

        return false;
    }
}
