<?php
    /**
     * Client library for Payssion API.
     */
    class Payssion
    {
        /**
         * @const string
         */
        const VERSION = '1.3.0.160612';

        /**
         * @var string
         */
        protected $api_url;
        protected $api_key = ''; //your api key
        protected $secret_key = ''; //your secret key
        protected static $sig_keys = array(
                'create' => array(
                        'api_key', 'pm_id', 'amount', 'currency', 'order_id', 'secret_key'
                ),
                'details' => array(
                        'api_key', 'transaction_id', 'order_id', 'secret_key'
                )
        );

        /**
         * @var array
         */
        protected $http_errors = array(
            400 => '400 Bad Request',
            401 => '401 Unauthorized',
            500 => '500 Internal Server Error',
            501 => '501 Not Implemented',
            502 => '502 Bad Gateway',
            503 => '503 Service Unavailable',
            504 => '504 Gateway Timeout',
        );
        /**
         * @var bool
         */
        protected $is_success = false;
        /**
         * @var array
         */
        protected $allowed_request_methods = array(
            'get',
            'put',
            'post',
            'delete',
        );

        /**
         * Constructor
         *
         * @param string $api_key Payssion App api_key
         * @param string $secret_key Payssion App secret_key
         * @param bool $is_livemode false if you use sandbox api_key and true for live mode
         */
        public function __construct($api_key, $secret_key, $is_livemode = true)
        {
            $this->api_key = $api_key;
            $this->secret_key = $secret_key;

            $validate_params = array(
                false === extension_loaded('curl') => 'The curl extension must be loaded for using this class!',
                false === extension_loaded('json') => 'The json extension must be loaded for using this class!',
                empty($this->api_key) => 'api_key is not set!',
                empty($this->secret_key) => 'secret_key is not set!',
            );
            $this->checkForErrors($validate_params);

            $this->setLiveMode($is_livemode);
        }
        /**
         * Set LiveMode
         *
         * @param bool $is_livemode
         */
        public function setLiveMode($is_livemode)
        {
            if ($is_livemode) {
                $this->api_url = 'https://www.payssion.com/api/v1/payment/';
            } else {
                $this->api_url = 'http://sandbox.payssion.com/api/v1/payment/';
            }
        }
        /**
         * Set Api URL
         *
         * @param string $url Api URL
         */
        public function setUrl($url)
        {
            $this->api_url = $url;
        }

        /**
         * Request state getter
         *
         * @return bool
         */
        public function isSuccess()
        {
            return $this->is_success;
        }
        /**
         * create payment order
         *
         * @param $params create Params
         * @return array
         */
        public function create(array $params)
        {
            return $this->call(
                'create',
                'post',
                 $params
            );
        }

        /**
         * get payment details
         *
         * @param $params query Params
         * @return array
         */
        public function getDetails(array $params)
        {
            return $this->call(
                    'details',
                    'post',
                    $params
            );
        }
        /**
         * Method responsible for preparing, setting state and returning answer from rest server
         *
         * @param string $method
         * @param string $request
         * @param array $params
         * @return array
         */
        protected function call($method, $request, $params)
        {
            $this->is_success = false;

            $validate_params = array(
                false === is_string($method) => 'Method namemust be string',
                false === $this->checkRequestMethod($request) => 'Not allowed request method type',
                true === empty($params) => 'params is null',
            );
            $this->checkForErrors($validate_params);

            $params['api_key'] = $this->api_key;
            $params['secret_key'] = $this->secret_key;
            $params['api_sig'] = $this->getSig($params, self::$sig_keys[$method]);

            $rq = new CURLRequester($this->api_url . $method, $request, true, true);
            foreach ($this->getHeaders() as $header) {
                $rq->addHeader($header, null, true);
            }
            $response = $rq->request($params);

            if (isset($response['result_code']) && 200 == $response['result_code']) {
                $this->is_success = true;
            }
            return $response;
        }
        /**
         * Checking error mechanism
         *
         * @param array $validateArray
         * @throws Exception
         */
        protected function getSig(array &$params, array $sig_keys)
        {
            $msg_array = array();
            foreach ($sig_keys as $key) {
                $msg_array[$key] = isset($params[$key]) ? $params[$key] : '';
            }
            $msg_array['secret_key'] = $this->secret_key;

            $msg = implode('|', $msg_array);
            $sig = md5($msg);
            return $sig;
        }

        /**
         * Checking error mechanism
         *
         * @param array $validateArray
         * @throws Exception
         */
        protected function checkForErrors(&$validate_params)
        {
            foreach ($validate_params as $key => $error) {
                if ($key) {
                    throw new Exception($error, -1);
                }
            }
        }
        /**
         * Check if method is allowed
         *
         * @param string $method_type
         * @return bool
         */
        protected function checkRequestMethod($method_type)
        {
            $request_method = strtolower($method_type);
            if (in_array($request_method, $this->allowed_request_methods)) {
                return true;
            }
            return false;
        }

        protected function &getHeaders()
        {
            $langVersion = phpversion();
            $uname= php_uname();
            $ua = array(
                    'version' => self::VERSION,
                    'lang' => 'php',
                    'lang_version' => $langVersion,
                    'publisher' => 'payssion and Jakob M.',
                    'application' => 'GExtension by ibot3',
                    'uname' => $uname,
            );
            $headers = array(
                    'X-Payssion-Client-User-Agent: ' . json_encode($ua),
                    "User-Agent: Payssion/php/$langVersion/" . self::VERSION,
                    'Content-Type: application/x-www-form-urlencoded',
            );

            return $headers;
        }

        public static function GetPaymentMethods($all = false)
        {
            //https://shancarter.github.io/mr-data-converter/
            //http://php.fnlist.com/php/json_decode

            $methods_f = array();
            $methods_uf = array(
                'fpx_my' =>
                array(
                'name' => 'Myclear FPX',
                'country' => 'Malaysia',
                ),
                'hlb_my' =>
                array(
                'name' => 'Hong Leong',
                'country' => 'Malaysia',
                ),
                'maybank2u_my' =>
                array(
                'name' => 'Maybank2u',
                'country' => 'Malaysia',
                ),
                'cimb_my' =>
                array(
                'name' => 'CIMB Clicks',
                'country' => 'Malaysia',
                ),
                'affinepg_my' =>
                array(
                'name' => 'Affin Bank',
                'country' => 'Malaysia',
                ),
                'amb_my' =>
                array(
                'name' => 'Am online',
                'country' => 'Malaysia',
                ),
                'rhb_my' =>
                array(
                'name' => 'RHB Now',
                'country' => 'Malaysia',
                ),
                'molwallet_my' =>
                array(
                'name' => 'MOLWallet',
                'country' => 'Malaysia',
                ),
                'webcash_my' =>
                array(
                'name' => 'Webcash',
                'country' => 'Malaysia',
                ),
                '7eleven_my' =>
                array(
                'name' => '7-eleven',
                'country' => 'Malaysia',
                ),
                'esapay_my' =>
                array(
                'name' => 'Esapay',
                'country' => 'Malaysia',
                ),
                'epay_my' =>
                array(
                'name' => 'epay',
                'country' => 'Malaysia',
                ),
                'enets_sg' =>
                array(
                'name' => 'eNets',
                'country' => 'Singapore',
                ),
                'singpost_sg' =>
                array(
                'name' => 'SAM by SingPost',
                'country' => 'Singapore',
                ),
                'atmva_id' =>
                array(
                'name' => 'ATMVA',
                'country' => 'Indonesia',
                ),
                'nganluong_vn' =>
                array(
                'name' => 'Nganluong',
                'country' => 'Vietnam',
                ),
                'vtcpay_vn' =>
                array(
                'name' => 'VTCPay',
                'country' => 'Vietnam',
                ),
                'viettel_vn' =>
                array(
                'name' => 'Viettel',
                'country' => 'Vietnam',
                ),
                'vinaphone_vn' =>
                array(
                'name' => 'Vinaphone',
                'country' => 'Vietnam',
                ),
                'mobifone_vn' =>
                array(
                'name' => 'MobiFone',
                'country' => 'Vietnam',
                ),
                'dragonpay_ph' =>
                array(
                'name' => 'Dragonpay',
                'country' => 'Philippines',
                ),
                'bayad_ph' =>
                array(
                'name' => 'Bayad Center',
                'country' => 'Philippines',
                ),
                'cebuana_ph' =>
                array(
                'name' => 'Cebuana Lhuillier',
                'country' => 'Philippines',
                ),
                'unionpay_ph' =>
                array(
                'name' => 'China UnionPay',
                'country' => 'Philippines',
                ),
                'ecpay_ph' =>
                array(
                'name' => 'ECPay',
                'country' => 'Philippines',
                ),
                'gcash_ph' =>
                array(
                'name' => 'Globe Gcash',
                'country' => 'Philippines',
                ),
                'lbc_ph' =>
                array(
                'name' => 'LBC',
                'country' => 'Philippines',
                ),
                'mlh_ph' =>
                array(
                'name' => 'M. Lhuillier',
                'country' => 'Philippines',
                ),
                'rds_ph' =>
                array(
                'name' => 'Robinsons Dept Store',
                'country' => 'Philippines',
                ),
                'smr_ph' =>
                array(
                'name' => 'SM Payment Counters',
                'country' => 'Philippines',
                ),
                'bol_ph' =>
                array(
                'name' => 'Bancnet Online',
                'country' => 'Philippines',
                ),
                '711_ph' =>
                array(
                'name' => '7-Eleven',
                'country' => 'Philippines',
                ),
                'cherrycredits' =>
                array(
                'name' => 'CherryCredits',
                'country' => 'Global including South East',
                ),
                'molpoints' =>
                array(
                'name' => 'MOLPoints',
                'country' => 'Global including South East',
                ),
                'molpointscard' =>
                array(
                'name' => 'MOLPoints card',
                'country' => 'Global including South East',
                ),
                'alipay_cn' =>
                array(
                'name' => 'Alipay',
                'country' => 'China',
                ),
                'tenpay_cn' =>
                array(
                'name' => 'Tenpay',
                'country' => 'China',
                ),
                'unionpay_cn' =>
                array(
                'name' => 'Unionpay',
                'country' => 'China',
                ),
                'creditcard_jp' =>
                array(
                'name' => 'Credit Card Japan',
                'country' => 'Japan',
                ),
                'docomo_jp' =>
                array(
                'name' => 'Docomo Carrier Billing',
                'country' => 'Japan',
                ),
                'webmoney_jp' =>
                array(
                'name' => 'WebMoney Japan',
                'country' => 'Japan',
                ),
                'bitcash_jp' =>
                array(
                'name' => 'Bitcash',
                'country' => 'Japan',
                ),
                'netcash_jp' =>
                array(
                'name' => 'Netcash',
                'country' => 'Japan',
                ),
                'cashu' =>
                array(
                'name' => 'cashU',
                'country' => 'Middle East & North Africa',
                ),
                'onecard' =>
                array(
                'name' => 'onecard',
                'country' => 'Middle East & North Africa',
                ),
                'paybyme_tr' =>
                array(
                'name' => 'pabyme visa',
                'country' => 'Turkey',
                ),
                'ttnet_tr' =>
                array(
                'name' => 'TTNET ÖdemeT',
                'country' => 'Turkey',
                ),
                'dineromail_ar' =>
                array(
                'name' => 'dineromail',
                'country' => 'Argentina',
                ),
                'santander_ar' =>
                array(
                'name' => 'Santander Rio',
                'country' => 'Argentina',
                ),
                'link_ar' =>
                array(
                'name' => 'Red Link',
                'country' => 'Argentina',
                ),
                'pagofacil_ar' =>
                array(
                'name' => 'Pago Fácil',
                'country' => 'Argentina',
                ),
                'dmtransfer_ar' =>
                array(
                'name' => 'Dinero Mail (Transfer)',
                'country' => 'Argentina',
                ),
                'dmcash_ar' =>
                array(
                'name' => 'Dinero Mail (Cash)',
                'country' => 'Argentina',
                ),
                'rapipago_ar' =>
                array(
                'name' => 'Rapi Pago',
                'country' => 'Argentina',
                ),
                'naranja_ar' =>
                array(
                'name' => 'Tarjeta Naranja',
                'country' => 'Argentina',
                ),
                'arjetashopping_ar' =>
                array(
                'name' => 'Tarjeta Shopping',
                'country' => 'Argentina',
                ),
                'nativa_ar' =>
                array(
                'name' => 'Nativa',
                'country' => 'Argentina',
                ),
                'cencosud_ar' =>
                array(
                'name' => 'Cencosud',
                'country' => 'Argentina',
                ),
                'cabal_ar' =>
                array(
                'name' => 'Cabal',
                'country' => 'Argentina',
                ),
                'argencard_ar' =>
                array(
                'name' => 'Argencard',
                'country' => 'Argentina',
                ),
                'visadebit_ar' =>
                array(
                'name' => 'Visa Debit Argentina',
                'country' => 'Argentina',
                ),
                'mastercarddebit_ar' =>
                array(
                'name' => 'MasterCard Debit Argentina',
                'country' => 'Argentina',
                ),
                'bancodobrasil_br' =>
                array(
                'name' => 'bancodobrasil',
                'country' => 'Brazil',
                ),
                'itau_br' =>
                array(
                'name' => 'itau',
                'country' => 'Brazil',
                ),
                'boleto_br' =>
                array(
                'name' => 'Boleto',
                'country' => 'Brazil',
                ),
                'mercadolibre_br' =>
                array(
                'name' => 'Cartao MercadoLivre',
                'country' => 'Brazil',
                ),
                'bradesco_br' =>
                array(
                'name' => 'bradesco',
                'country' => 'Brazil',
                ),
                'hsbc_br' =>
                array(
                'name' => 'hsbc',
                'country' => 'Brazil',
                ),
                'caixa_br' =>
                array(
                'name' => 'caixa',
                'country' => 'Brazil',
                ),
                'santander_br' =>
                array(
                'name' => 'Santander',
                'country' => 'Brazil',
                ),
                'visa_br' =>
                array(
                'name' => 'visa',
                'country' => 'Brazil',
                ),
                'mastercard_br' =>
                array(
                'name' => 'mastercard',
                'country' => 'Brazil',
                ),
                'dinersclub_br' =>
                array(
                'name' => 'dinersclub',
                'country' => 'Brazil',
                ),
                'americanexpress_br' =>
                array(
                'name' => 'americanexpress',
                'country' => 'Brazil',
                ),
                'elo_br' =>
                array(
                'name' => 'elo',
                'country' => 'Brazil',
                ),
                'hipercard_br' =>
                array(
                'name' => 'hipercard',
                'country' => 'Brazil',
                ),
                'bancomer_mx' =>
                array(
                'name' => 'bancomer',
                'country' => 'Mexico',
                ),
                'banamex_mx' =>
                array(
                'name' => 'banamex',
                'country' => 'Mexico',
                ),
                'santander_mx' =>
                array(
                'name' => 'santander',
                'country' => 'Mexico',
                ),
                'oxxo_mx' =>
                array(
                'name' => 'oxxo',
                'country' => 'Mexico',
                ),
                'debitcard_mx' =>
                array(
                'name' => 'debit card (visa or mastercard)',
                'country' => 'Mexico',
                ),
                'redpagos_uy' =>
                array(
                'name' => 'redpagos',
                'country' => 'Uruguay',
                ),
                'bancochile_cl' =>
                array(
                'name' => 'Banco de Chile',
                'country' => 'Chile',
                ),
                'redcompra_cl' =>
                array(
                'name' => 'RedCompra',
                'country' => 'Chile',
                ),
                'webpay_cl' =>
                array(
                'name' => 'WebPay plus',
                'country' => 'Chile',
                ),
                'servipag_cl' =>
                array(
                'name' => 'Servipag',
                'country' => 'Chile',
                ),
                'presto_cl' =>
                array(
                'name' => 'Presto',
                'country' => 'Chile',
                ),
                'cmr_cl' =>
                array(
                'name' => 'CMR',
                'country' => 'Chile',
                ),
                'magna_cl' =>
                array(
                'name' => 'Magna',
                'country' => 'Chile',
                ),
                'santander_cl' =>
                array(
                'name' => 'Santander',
                'country' => 'Chile',
                ),
                'efecty_co' =>
                array(
                'name' => 'Efecty',
                'country' => 'Colombia',
                ),
                'pse_co' =>
                array(
                'name' => 'PSE',
                'country' => 'Colombia',
                ),
                'davivienda_co' =>
                array(
                'name' => 'Davivienda',
                'country' => 'Colombia',
                ),
                'exito_co' =>
                array(
                'name' => 'Almacenes Exito',
                'country' => 'Colombia',
                ),
                'baloto_co' =>
                array(
                'name' => 'Baloto',
                'country' => 'Colombia',
                ),
                'oc_co' =>
                array(
                'name' => 'Banco de Occidente',
                'country' => 'Colombia',
                ),
                'carulla_co' =>
                array(
                'name' => 'Carulla',
                'country' => 'Colombia',
                ),
                'edeq_co' =>
                array(
                'name' => 'EDEQ',
                'country' => 'Colombia',
                ),
                'surtimax_co' =>
                array(
                'name' => 'Surtimax',
                'country' => 'Colombia',
                ),
                'visa_co' =>
                array(
                'name' => 'Visa Colombia',
                'country' => 'Colombia',
                ),
                'visadebit_co' =>
                array(
                'name' => 'Visa Colombia',
                'country' => 'Colombia',
                ),
                'mastercard_co' =>
                array(
                'name' => 'MasterCard Colombia',
                'country' => 'Colombia',
                ),
                'mastercarddebit_co' =>
                array(
                'name' => 'MasterCard Debit Colombia',
                'country' => 'Colombia',
                ),
                'dinersclub_co' =>
                array(
                'name' => 'Diners Club Colombia',
                'country' => 'Colombia',
                ),
                'americanexpress_co' =>
                array(
                'name' => 'American Express Colombia',
                'country' => 'Colombia',
                ),
                'qiwi' =>
                array(
                'name' => 'QIWI',
                'country' => 'Global',
                ),
                'yamoney' =>
                array(
                'name' => 'Yandex.Money',
                'country' => 'Global',
                ),
                'yamoneyac' =>
                array(
                'name' => 'Bank Card (Yandex.Money)',
                'country' => 'Russia',
                ),
                'yamoneygp' =>
                array(
                'name' => 'Cash (Yandex.Money)',
                'country' => 'Russia',
                ),
                'moneta_ru' =>
                array(
                'name' => 'Moneta',
                'country' => 'Russia',
                ),
                'sberbank_ru' =>
                array(
                'name' => 'Sberbank',
                'country' => 'Russia',
                ),
                'alfaclick_ru' =>
                array(
                'name' => 'Alfa-Click',
                'country' => 'Russia',
                ),
                'promsvyazbank_ru' =>
                array(
                'name' => 'Promsvyazbank',
                'country' => 'Russia',
                ),
                'faktura_ru' =>
                array(
                'name' => 'Faktura',
                'country' => 'Russia',
                ),
                'russianpost_ru' =>
                array(
                'name' => 'Russian Post centres',
                'country' => 'Russia',
                ),
                'banktransfer_ru' =>
                array(
                'name' => 'Russia Bank transfer',
                'country' => 'Russia',
                ),
                'euroset_ru' =>
                array(
                'name' => 'Euroset',
                'country' => 'Russia',
                ),
                'beeline_ru' =>
                array(
                'name' => 'Beeline',
                'country' => 'Russia',
                ),
                'megafon_ru' =>
                array(
                'name' => 'Megafon',
                'country' => 'Russia',
                ),
                'mtc_ru' =>
                array(
                'name' => 'MTC',
                'country' => 'Russia',
                ),
                'tele2_ru' =>
                array(
                'name' => 'Tele2',
                'country' => 'Russia',
                ),
                'paysafecard' =>
                array(
                'name' => 'Paysafecard',
                'country' => 'Global',
                ),
                'sofort' =>
                array(
                'name' => 'Sofort',
                'country' => 'Europe',
                ),
                'trustpay' =>
                array(
                'name' => 'Trustpay',
                'country' => 'Europe',
                ),
                'giropay_de' =>
                array(
                'name' => 'Giropay',
                'country' => 'Germany',
                ),
                'eps_at' =>
                array(
                'name' => 'EPS',
                'country' => 'Austria',
                ),
                'bancontact_be' =>
                array(
                'name' => 'Bancontact/Mistercash',
                'country' => 'Belgium',
                ),/*
                'p24_pl' =>
                array(
                'name' => 'P24',
                'country' => 'Poland',
                ),*/
                'ideal_nl' =>
                array(
                'name' => 'iDeal',
                'country' => 'Netherlands',
                ),
                'teleingreso_es' =>
                array(
                'name' => 'Teleingreso',
                'country' => 'Spain',
                ),
                'multibanco_pt' =>
                array(
                'name' => 'Multibanco',
                'country' => 'Portugal',
                ),
                'neosurf' =>
                array(
                'name' => 'Neosurf',
                'country' => 'France',
                ),
                'polipayment' =>
                array(
                'name' => 'Polipayment',
                'country' => 'Australia & New Zealand',
                ),
                'openbucks' =>
                array(
                'name' => 'openbucks',
                'country' => 'North America',
                ),
                'bitcoin' =>
                array(
                'name' => 'bitcoin, litecoin…',
                'country' => 'Global',
                ),
                'dotpay_pl' =>
                array(
                    'name' => 'Dotpay',
                    'country' => 'Poland'
                )
            );

            foreach ($methods_uf as $id => $method) {
                $country = $method['country'];

                if (!isset($methods_f[$country])) {
                    $methods_f[$country] = array();
                }

                $methods_f[$country][$id] = $method['name'];
            }

            ksort($methods_f);

            foreach ($methods_f as $country => $methods) {
                ksort($methods_f[$country]);
            }

            $global = array('Global' => $methods_f['Global']);
            unset($methods_f['Global']);
            $methods_f = array_merge($global, $methods_f);

            if (!$all) {
                $enabled = FromJson(Settings::Get("settings_donations_payssion_methods"));

                foreach ($methods_f as $country => $c_methods) {
                    foreach ($c_methods as $id => $name) {
                        if (!in_array($id, $enabled)) {
                            unset($methods_f[$country][$id]);
                        }
                    }

                    if (empty($methods_f[$country])) {
                        unset($methods_f[$country]);
                    }
                }

                return $methods_f;
            } else {
                return $methods_f;
            }
        }

        public static function PostProcess($POST)
        {
            $db = MysqliDb::getInstance();

            Debug('Payssion_IPN: ' . ToJson($POST));

            $pm_id = $POST['pm_id'];
            $transaction_id = $POST['transaction_id'];
            $amount = FormatPrice($POST['amount']);
            $currency = $POST['currency'];
            $order_id = $POST['order_id'];
            $state = $POST['state'];
            $check_array = array(
                    Settings::Get('settings_donations_payssion_apikey'),
                    $pm_id,
                    $amount,
                    $currency,
                    $order_id,
                    $state,
                    Settings::Get('settings_donations_payssion_secretkey')
            );

            $check_msg = implode('|', $check_array);
            $check_sig = md5($check_msg);
            $notify_sig = $_POST['notify_sig'];

            if ($notify_sig == $check_sig) {
                if (isset($order_id)) {
                    $payment = Payment::PostProcess($order_id);

                    if ($payment) {
                        Debug('Payment found. Payment Hash: ' . $order_id);

                        $buyer = new User($payment['buyer']);
                        $for = new User($payment['for']);
                        $coupon = new Coupon($payment['coupon']);
                        $price = $payment['price'];
                        $package = new Package($payment['package'], null, true, $buyer);

                        if ($buyer->valid && $for->valid && $package->valid && $price) {
                            if (Payment::IsProblematic('payssion', $state)) {
                                $db->where("gateway", "payssion");
                                $db->where("transactionid", $transaction_id);

                                $data = $db->getOne("donations");

                                if ($db->count) {
                                    $donation = new Donation($data['id'], $data);

                                    if (!$donation->GetValue('revoked')) {
                                        $donation->Revoke();

                                        foreach (Permissions::GetSteamIDsWithPermission(true, 'admin_donations') as $steamid64) {
                                            Notifications::Send($steamid64, 'notification_donation_revoked_payssion', array('donationid' => $donation["id"], 'amount' => Settings::Get('settings_donations_currency') . $donation["amount"]), 'index.php?t=admin_donations&id=' . $donation["id"], 'star');
                                        }
                                    }
                                }
                            } elseif (Payment::Validate('payssion', array($amount, $currency, $state, $package, $price, $POST))) {
                                Debug('Payment verified.');

                                $db->where("transactionid", $transaction_id);
                                $db->getOne("donations");

                                if ($db->count) {
                                    Error('TransactionID already proceeded', true);
                                }

                                if ($payment['coupon']) {
                                    if ($coupon->valid) {
                                        $coupon->Devaluate();
                                    } else {
                                        Payment::CouponRefund($for, $transaction_id, $amount);
                                        Error('Invalid coupon. Refunding...', true);
                                    }
                                }

                                $data = array(
                                    'name_buyer' => $for->GetValue('nick'),
                                    'steamid64_buyer' => $buyer->GetValue('steamid64'),
                                    'steamid64_for' => $for->GetValue('steamid64'),
                                    'amount' => $amount,
                                    'package' => $package->GetValue('id'),
                                    'transactionid' => $transaction_id,
                                    'coupon' => $payment['coupon'],
                                    'email' => '',
                                    'gateway' => 'payssion'
                                );

                                Payment::Finish($data, $buyer, $for, $package, $payment);
                            }
                        }
                    }
                }
            }
        }
    }
