<?php
    class Stripe
    {
        private $apiurl = 'https://api.stripe.com/v1/';

        public function MakeCharge($token, $amount, $description)
        {
            $requester = new CURLRequester($this->apiurl . 'charges', 'POST', true, true);
            $response = $requester->request(array('amount' => $this->GetPrice($amount), 'currency' => Settings::Get('settings_donations_currency_plain'), 'description' => $description, 'source' => $token), false, 5, false, Settings::Get('settings_donations_stripe_secretkey') . ':');

            return $response;
        }

        public static function GetPrice($price)
        {
            $zeroDecimalCurrencies = array('MGA', 'BIF', 'CLP', 'PYG', 'DJF', 'RWF', 'GNF', 'JPY', 'VND', 'VUV', 'XAF', 'KMF', 'KRW', 'XOF', 'XPF');

            if (!in_array(Settings::Get('settings_donations_currency_plain'), $zeroDecimalCurrencies)) {
                $price = $price * 100;
            }

            return intval($price);
        }

        public static function MakePayment($paymentIntentId, $price, $paymentMethodId, $useStripeSdk, $paymentId){
            \Stripe\Stripe::setApiKey(Settings::Get('settings_donations_stripe_secretkey'));

            try {
                if($paymentMethodId != null) {
                    $payment = Payment::PostProcess($paymentId);

                    // Create new PaymentIntent with a PaymentMethod ID from the client.
                    $intent = \Stripe\PaymentIntent::create([
                        "amount" => Stripe::GetPrice($price),
                        "currency" => Settings::Get('settings_donations_currency_plain'),
                        "payment_method" => $paymentMethodId,
                        "confirmation_method" => "manual",
                        "confirm" => true,
                        // If a mobile client passes `useStripeSdk`, set `use_stripe_sdk=true`
                        // to take advantage of new authentication features in mobile SDKs
                        "use_stripe_sdk" => $useStripeSdk,
                        "metadata" => array("gex_payment_id" => $paymentId),
                        'description' => $payment['package_name']
                    ]);

                    if($intent->status == \Stripe\PaymentIntent::STATUS_SUCCEEDED){
                        Debug("Stripe payment complete. gexPaymentID: " . $paymentId);
                        Stripe::FinishPayment($intent);
                    }
                    // After create, if the PaymentIntent's status is succeeded, fulfill the order.
                } else if ($paymentIntentId != null) {
                    // Confirm the PaymentIntent to finalize payment after handling a required action
                    // on the client.
                    $intent = \Stripe\PaymentIntent::retrieve($paymentIntentId);
                    $intent->confirm();

                    if($intent->status == \Stripe\PaymentIntent::STATUS_SUCCEEDED){
                        Debug("Stripe payment complete. gexPaymentID: " . $intent->metadata->gex_payment_id);
                        Stripe::FinishPayment($intent);
                    }

                    // After confirm, if the PaymentIntent's status is succeeded, fulfill the order.
                }

                return $intent;
            } catch (\Stripe\Error\Card $e) {
                Debug("Stripe Error: " . $e->getMessage());
            }

            return False;
        }

        public static function GenerateResponse($intent){
            \Stripe\Stripe::setApiKey(Settings::Get('settings_donations_stripe_secretkey'));

            switch($intent->status) {
                case "requires_action":
                case "requires_source_action":
                    // Card requires authentication
                    return [
                        'requiresAction'=> true,
                        'paymentIntentId'=> $intent->id,
                        'clientSecret'=> $intent->client_secret
                    ];
                case "requires_payment_method":
                case "requires_source":
                    // Card was not properly authenticated, suggest a new payment method
                    return [
                        error => "Your card was denied, please provide a new payment method"
                    ];
                case "succeeded":
                    // Payment is complete, authentication not required
                    // To cancel the payment after capture you will need to issue a Refund (https://stripe.com/docs/api/refunds)
                    return ['clientSecret' => $intent->client_secret];
            }
        }

        public static function FinishPayment($intent){
            global $db;

            $paymentId = $intent->metadata->gex_payment_id;

            $payment = Payment::PostProcess($paymentId);

            if ($payment) {
                $buyer = new User($payment['buyer']);
                $for = new User($payment['for']);
                $coupon = new Coupon($payment['coupon']);
                $price = $payment['price'];
                $package = new Package($payment['package'], null, true, $buyer);

                if ($buyer->valid && $for->valid && $package->valid && $price) {
                    if ($package->Buyable($for)) {
                        if (!$payment['coupon'] || $coupon->valid) {
                            $coupon->Devaluate();

                            $db->where('transactionid', $intent['id']);
                            $db->getOne('donations');

                            if ($db->count) {
                                Error('TransactionID already proceeded', true);
                            }

                            $name = $buyer->GetValue('nick');

                            $data = array(
                                'name_buyer' => $name,
                                'steamid64_buyer' => $buyer->GetValue('steamid64'),
                                'steamid64_for' => $for->GetValue('steamid64'),
                                'amount' => $price,
                                'package' => $package->GetValue('id'),
                                'transactionid' => $intent['id'],
                                'coupon' => $payment['coupon'],
                                'email' => $buyer->GetValue('email'),
                                'gateway' => 'stripe',
                            );

                            Payment::Finish($data, $buyer, $for, $package, $payment);

                            Debug("Stripe: Payment finished");
                        } else {
                            Debug('Stripe: Invalid coupon.');
                        }
                    } else {
                        Debug('Package not buyable.');
                    }
                }else{
                    Debug("Stripe: Invalid entity or price");
                }
            }else{
                Debug("Stripe: GEx payment not found.");
            }
        }
    }
