<?php

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

class User extends DBTableClass
{
    public $groups;
    public $statistics;

    public function __construct($id, $data = null)
    {
        if (StartsWith($id, 'STEAM')) {
            $id = SteamIDTo64($id);
        }

        $this->Initialize('users', $id, 'steamid64', $data, array('ips', 'groups'));

        if ($this->valid) {
            $this->statistics = new Statistics($this);
        }
    }

    public function GetGroup($bundle = 0)
    {
        if (!$this->groups) {
            if ($this->GetValue('groups')) {
                foreach ($this->GetValue('groups') as $bundlestring => $groupname) {
                    $parts = explode('_', $bundlestring);
                    if (isset($parts[1])) {
                        if (is_numeric($parts[1])) {
                            $group = new Group($groupname);

                            if ($group->valid) {
                                $this->groups[$parts[1]] = $group;
                            }
                        }
                    }
                }
            } else {
                $groups = array();
            }
        }

        if (isset($this->groups[$bundle])) {
            $group = $this->groups[$bundle];

            if ($group->valid) {
                return $group;
            }
        } else {
            return new Group(Settings::Get('settings_general_defaultgroup'));
        }
    }

    public function SetGroup($groupname, $bundle = 0, $force = false)
    {
        $newgroup = new Group($groupname);

        if ($newgroup->valid) {
            if ($newgroup->GetValue('groupname') != $this->GetGroup($bundle)->GetValue('groupname') || $force) {
                if (!tempty($this->GetValue('ts3uid')) && (!tempty($this->GetGroup($bundle)->GetTs3Servergroup($bundle)) || !tempty($newgroup->GetTs3Servergroup($bundle)))) {
                    $ts3worker = GetTS3Worker();

                    if ($ts3worker->valid) {
                        if (!tempty($this->GetGroup($bundle)->GetTs3Servergroup($bundle))) {
                            $ts3worker->RemoveServergroup($this->GetValue('ts3uid'), $this->GetGroup($bundle)->GetTs3Servergroup($bundle));
                        }

                        if (!tempty($newgroup->GetTs3Servergroup($bundle))) {
                            $ts3worker->AssignServergroup($this->GetValue('ts3uid'), $newgroup->GetTs3Servergroup($bundle));
                        }
                    }
                }

                if (!tempty($this->GetValue('discord_user_id')) && (!tempty($this->GetGroup($bundle)->GetDiscordRole($bundle)) || !tempty($newgroup->GetDiscordRole($bundle)))) {
                    $discord = new Discord();

                    if (!tempty($this->GetGroup($bundle)->GetDiscordRole($bundle))) {
                        $discord->removeRoleFromUser($this->GetValue('discord_user_id'), $this->GetGroup($bundle)->GetDiscordRole($bundle));
                    }

                    if (!tempty($newgroup->GetDiscordRole($bundle))) {
                        $discord->addRoleToUser($this->GetValue('discord_user_id'), $newgroup->GetDiscordRole($bundle));
                    }
                }

                $group_before = $this->GetGroup($bundle);

                $this->groups[$bundle] = $newgroup;

                $groups = $this->GetValue('groups');

                $groups['bundle_'.$bundle] = $groupname;

                $this->SetValue('groups', ToJson($groups, JSON_FORCE_OBJECT));

                if ($bundle > 0 && '' != $newgroup->GetValue('webgroup')) {
                    $webgroup = $this->GetHighestAssociatedWebGroup();

                    if (null !== $webgroup) {
                        $currentwebgroup = $this->GetGroup();

                        if (!$currentwebgroup->valid || ($webgroup->GetValue('level') > $currentwebgroup->GetValue('level') || $group_before->GetValue('webgroup') == $currentwebgroup->GetValue('groupname'))) {
                            $this->SetGroup($webgroup->GetValue('groupname'));
                        }
                    }
                }
            }
        }
    }

    public function GetHighestAssociatedWebGroup()
    {
        $highest = null;

        foreach (Serverbundle::GetAll() as $bundle) {
            $webgroup_name = $this->GetGroup($bundle->GetValue('id'))->GetValue('webgroup');

            if ('' != $webgroup_name) {
                $webgroup = new Group($webgroup_name);

                if ($webgroup->valid) {
                    if (null === $highest || $webgroup->GetValue('level') > $highest->GetValue('level')) {
                        $highest = $webgroup;
                    }
                }
            }
        }

        return $highest;
    }

    public function RefreshSteamData()
    {
        $steamprofile = SteamAPI::GetUserData($this->GetValue('steamid64'));
        if (isset($steamprofile['steamid64'])) {
            if ($steamprofile['steamid64'] == $this->GetValue('steamid64')) {
                $this->SetValue('nick', $steamprofile['personaname']);
                $this->SetValue('avatar_small', $steamprofile['avatar']);
                $this->SetValue('avatar_medium', $steamprofile['avatarmedium']);
                $this->SetValue('avatar_large', $steamprofile['avatarfull']);
                $this->SetValue('steamid32', $steamprofile['steamid32']);
                $this->RefreshData();

                return true;
            }
        }

        return false;
    }

    public function GetPackages($activeonly = false)
    {
        if ($activeonly) {
            $GLOBALS['db']->where('status = 1 AND expires > NOW()');
        }

        $GLOBALS['db']->where('steamid64', $this->GetValue('steamid64'));

        return $GLOBALS['db']->get('packages_bought');
    }

    public function GetBans()
    {
        $GLOBALS['db']->where('steamid64 = ? AND status < 2', array($this->GetValue('steamid64')));
        $GLOBALS['db']->orderBy('date_banned', 'Desc');

        $bans = $GLOBALS['db']->get('bans');

        return $bans;
    }

    public function GetActiveBans($bundle = -1, $notbundle = -1)
    {
        $GLOBALS['db']->where('steamid64 = ? AND status = 0 AND (((date_banned + INTERVAL length MINUTE) > NOW()) OR length = 0)', array($this->GetValue('steamid64')));

        if ($bundle >= 0) {
            $GLOBALS['db']->where('serverbundle', $bundle);
        }

        if ($notbundle >= 0) {
            $GLOBALS['db']->where('serverbundle', $notbundle, '<>');
        }

        $bans = $GLOBALS['db']->get('bans');

        return $bans;
    }

    public function IsBanned($bundle = -1)
    {
        $bans = null;

        if ($bundle >= 0) {
            $bans = $this->GetActiveBans($bundle, -1);
        } else {
            $bans = $this->GetActiveBans(-1, 0);
        }

        if (sizeof($bans) > 0) {
            return true;
        } else {
            return false;
        }
    }

    public function AddTs3Servergroups()
    {
        foreach (Serverbundle::GetAll(true) as $bundle) {
            $group = $this->GetGroup($bundle->GetValue('id'));
            if ($group->valid) {
                if (!tempty($this->GetValue('ts3uid')) && $group->GetTs3Servergroup($bundle->GetValue('id'))) {
                    $ts3worker = GetTS3Worker();

                    if ($ts3worker->valid) {
                        $ts3worker->AssignServergroup($this->GetValue('ts3uid'), $group->GetTs3Servergroup($bundle->GetValue('id')));
                    }

                    $ts3worker->Close();
                }
            }
        }
    }

    public function RemoveTs3Servergroups()
    {
        foreach (Serverbundle::GetAll(true) as $bundle) {
            $group = $this->GetGroup($bundle->GetValue('id'));

            if ($group->valid) {
                if (!tempty($this->GetValue('ts3uid')) && $group->GetTs3Servergroup($bundle->GetValue('id'))) {
                    $ts3worker = GetTS3Worker();

                    if ($ts3worker->valid) {
                        $ts3worker->RemoveServergroup($this->GetValue('ts3uid'), $group->GetTs3Servergroup($bundle->GetValue('id')));
                    }

                    $ts3worker->Close();
                }
            }
        }
    }

    public function SendEmail($subject, $body, $attachments = array(), $debug = false)
    {
        $errors = array();

        if (!tempty(Config::Get('mail_host'))) {
            try {
                $mail = new PHPMailer();

                if ('smtp' == Config::Get('mail_method')) {
                    $mail->IsSMTP();
                } else {
                    $mail->IsSendmail();
                }

                $mail->IsHTML(true);
                $mail->SMTPAuth = true;
                $mail->Host = Config::Get('mail_host');
                $mail->Port = Config::Get('mail_port');
                $mail->Timeout = 2;

                if ($debug)
                    $mail->SMTPDebug = 3;

                if ('ssl' == Config::Get('mail_secure')) {
                    $mail->SMTPSecure = 'ssl';
                } elseif ('tls' == Config::Get('mail_secure')) {
                    $mail->SMTPSecure = 'tls';
                }

                $mail->CharSet = 'utf-8';

                if (!tempty(Config::Get('mail_username'))) {
                    $mail->Username = Config::Get('mail_username');
                    $mail->Password = Config::Get('mail_password');
                }

                $mail->SetFrom(Config::Get('mail_from'), (tempty(Settings::Get('settings_general_community_name')) ? 'GMOD Web' : Settings::Get('settings_general_community_name')));
                $mail->Subject = $subject;
                $mail->Body = $body;
                $mail->AddAddress($this->GetValue('email'));

                foreach ($attachments as $attachment) {
                    $mail->addStringAttachment($attachment['content'], $attachment['name']);
                }

                if ($mail->Send()) {
                    return true;
                } else {
                    $errors[] = $mail->ErrorInfo;
                }
            } catch (phpmailerException $e) {
                $errors[] = $e->errorMessage();
            } catch (Exception $e) {
                $errors[] = $e->errorMessage();
            }

            Debug(ToJson($errors));
        }

        return false;
    }

    public function GetDonations()
    {
        $GLOBALS['db']->where('steamid64_for', $this->GetValue('steamid64'));
        $GLOBALS['db']->orderBy('date', 'Desc');

        return $GLOBALS['db']->get('donations');
    }

    public static function CheckForRewards()
    {
        $product = new Product('gex');

        if ($product->valid) {
            if ((strtotime($product->GetValue('checked')) > time() || (time() - strtotime($product->GetValue('checked'))) > 300) || isset($_GET['force'])) {
                $product->VerifyLicense();
                $product->RefreshAvailableVersion();
                $product->SetValue('checked', date('Y-m-d H:i:s'));
            }
        }
    }

    public static function GetCount()
    {
        $db = MysqliDb::getInstance();

        return $db->getValue('users', 'COUNT(*)');
    }

    public static function GetFromSteamIDs($steamids64)
    {
        $users = array();

        foreach ($steamids64 as $steamid64) {
            $user = new User($steamid64);

            if ($user->valid) {
                $users[] = $user;
            }
        }

        return $users;
    }

    public static function Create($steamid64)
    {
        $user = new User($steamid64);

        if (!$user->valid) {
            $db = MysqliDb::getInstance();

            $steamprofile = SteamAPI::GetUserData($steamid64);

            if ($steamprofile) {
                $steamid32 = $steamprofile['steamid32'];
                $nick = $steamprofile['personaname'];
                $avatar_small = $steamprofile['avatar'];
                $avatar_medium = $steamprofile['avatarmedium'];
                $avatar_large = $steamprofile['avatarfull'];

                $ips = array(GetClientIP());

                $groups = array();

                if (User::GetCount() <= 1) {
                    $groups['bundle_0'] = 'superadmin';
                }

                $data = array(
                        'steamid64' => $steamid64,
                        'steamid32' => $steamid32,
                        'uniqueid' => crc32('gm_'.$steamid32.'_gm'),
                        'nick' => $nick,
                        'groups' => json_encode($groups),
                        'avatar_small' => $avatar_small,
                        'avatar_medium' => $avatar_medium,
                        'avatar_large' => $avatar_large,
                        'ips' => json_encode($ips),
                        'random' => random_string(25),
                        'ts3uid' => '',
                        'discord_username' => '',
                        'discord_user_id' => '',
                        'emailnotifications' => 1,
                        'email' => '',
                        'language' => '',
                        'accepted' => 0,
                    );

                $db->insert('users', $data);

                return true;
            } else {
                Debug(json_encode(array('error' => 'SteamAPI Error')));

                return false;
            }
        } else {
            return true;
        }
    }

    public static function GetByValue($key, $value, $like = false)
    {
        $db = MysqliDb::getInstance();

        if ($like) {
            $db->where($key, $value, 'like');
        } else {
            $db->where($key, $value);
        }

        $data = $db->get('users');

        $users = array();

        foreach ($data as $d) {
            $user = new User($d['steamid64'], $d);

            if ($user->valid) {
                $users[] = $user;
            }
        }

        return $users;
    }

    public function Unban($bundle = -1)
    {
        $db = MysqliDb::getInstance();

        global $auth_user;

        if (!$auth_user) {
            $auth_user = new User('0');
        }

        $active_bans = $this->GetActiveBans($bundle);

        foreach ($active_bans as $ban) {
            Bans::Unban($ban['id']);
        }
    }

    public function UpdateDiscordUsername($username)
    {
        global $db;

        if (empty($username)) {
            $this->RemoveDiscordRoles();

            $this->SetValue('discord_username', '');
            $this->SetValue('discord_user_id', '');

            return true;
        }

        $discord = new Discord();

        $d_user = $discord->getUserByUsername($username);

        if ($d_user && isset($d_user['id'])) {
            $db->where('discord_user_id', $d_user['id']);
            $taken_user = $db->getOne('users');

            if ($db->count) {
                return sprintf('Username already linked to %s', $taken_user['steamid64']);
            }

            $this->RemoveDiscordRoles();

            $this->SetValue('discord_username', $username);
            $this->SetValue('discord_user_id', $d_user['id']);

            $this->AddDiscordRoles();

            return true;
        } else {
            return 'Username not found in guild';
        }
    }

    public function AddDiscordRoles()
    {
        foreach (Serverbundle::GetAll(true) as $bundle) {
            $group = $this->GetGroup($bundle->GetValue('id'));
            if ($group->valid) {
                if (!tempty($this->GetValue('discord_user_id')) && $group->GetDiscordRole($bundle->GetValue('id'))) {
                    $discord = new Discord();

                    $discord->addRoleToUser($this->GetValue('discord_user_id'), $group->GetDiscordRole($bundle->GetValue('id')));
                }
            }
        }
    }

    public function RemoveDiscordRoles()
    {
        foreach (Serverbundle::GetAll(true) as $bundle) {
            $group = $this->GetGroup($bundle->GetValue('id'));
            if ($group->valid) {
                if (!tempty($this->GetValue('discord_user_id')) && $group->GetDiscordRole($bundle->GetValue('id'))) {
                    $discord = new Discord();

                    $discord->removeRoleFromUser($this->GetValue('discord_user_id'), $group->GetDiscordRole($bundle->GetValue('id')));
                }
            }
        }
    }

    public function RefreshCountryCode(){
        $ipinfo = GetIPInfo();

        if($ipinfo !== null && $ipinfo['country_code'] && $ipinfo['country_code'] != $this->GetValue('country_code')){
            $this->SetValue('country_code', $ipinfo['country_code']);
        }

    }

    public function GetOrRefreshCountryCode(){
        if($this->GetValue('country_code') !== null){
            return $this->GetValue('country_code');
        }else{
            $this->RefreshCountryCode();

            return $this->GetValue('country_code');
        }
    }
}
